<?php

namespace App\Console\Commands;

use App\Models\CompetitorBrand;
use App\Services\CompetitorDataFetchService;
use App\Services\CompetitorAlertService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SyncCompetitorData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'competitors:sync 
                            {--competitor= : Specific competitor ID to sync}
                            {--days=7 : Number of days to fetch data for}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync competitor social media data from all platforms';

    protected CompetitorDataFetchService $dataFetchService;
    protected CompetitorAlertService $alertService;

    /**
     * Create a new command instance.
     */
    public function __construct(
        CompetitorDataFetchService $dataFetchService,
        CompetitorAlertService $alertService
    ) {
        parent::__construct();
        $this->dataFetchService = $dataFetchService;
        $this->alertService = $alertService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting competitor data sync...');

        $competitorId = $this->option('competitor');
        $days = (int) $this->option('days');

        if ($competitorId) {
            $competitors = CompetitorBrand::where('id', $competitorId)
                ->active()
                ->get();
        } else {
            $competitors = CompetitorBrand::active()->get();
        }

        if ($competitors->isEmpty()) {
            $this->warn('No active competitors found to sync.');
            return 0;
        }

        $this->info("Found {$competitors->count()} competitor(s) to sync.");

        $bar = $this->output->createProgressBar($competitors->count());
        $bar->start();

        $successCount = 0;
        $errorCount = 0;

        foreach ($competitors as $competitor) {
            try {
                $this->line("\nSyncing: {$competitor->name}");
                
                // Fetch competitor posts
                $results = $this->dataFetchService->fetchCompetitorPosts(
                    $competitor,
                    now()->subDays($days)
                );

                // Log results
                foreach ($results as $platform => $result) {
                    if (isset($result['error'])) {
                        $this->error("  {$platform}: {$result['error']}");
                    } else {
                        $this->info("  {$platform}: {$result['stored']} new, {$result['updated']} updated");
                    }
                }

                // Check for alerts
                $alerts = $this->alertService->checkForAlerts($competitor);
                if (!empty($alerts)) {
                    $this->warn("  Generated {count($alerts)} alert(s)");
                }

                $successCount++;
            } catch (\Exception $e) {
                $this->error("  Failed: {$e->getMessage()}");
                Log::error("Failed to sync competitor {$competitor->id}: {$e->getMessage()}");
                $errorCount++;
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        $this->info("Sync completed!");
        $this->info("Success: {$successCount}");
        if ($errorCount > 0) {
            $this->warn("Errors: {$errorCount}");
        }

        return 0;
    }
}