<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use App\Models\AdVariant;
use App\Models\AdPerformanceMetric;
use App\Models\Campaign;
use App\Services\AdCopyGeneratorService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class AdManagerController extends Controller
{
    /**
     * Display a listing of ads.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // Guard clause for brand not found
        $brand = Auth::user()->brand;
        if (!$brand) {
            return response()->view('errors.brand-not-found', [], 404);
        }

        $ads = Ad::where('brand_id', $brand->id)
            ->with(['campaign', 'variants'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('brand.ads.index', compact('ads'));
    }

    /**
     * Show the form for creating a new ad.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Guard clause for brand not found
        $brand = Auth::user()->brand;
        if (!$brand) {
            return response()->view('errors.brand-not-found', [], 404);
        }

        $campaigns = Campaign::where('brand_id', $brand->id)
            ->orderBy('name')
            ->get();

        return view('brand.ads.create', compact('campaigns'));
    }

    /**
     * Store a newly created ad and generate variants.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Guard clause for brand not found
        $brand = Auth::user()->brand;
        if (!$brand) {
            return response()->view('errors.brand-not-found', [], 404);
        }

        $validator = Validator::make($request->all(), [
            'campaign_id' => 'required|exists:campaigns,id,brand_id,' . $brand->id,
            'name' => 'required|string|max:255',
            'product_info' => 'required|string',
            'audience' => 'required|string',
            'tone' => 'required|string',
            'budget' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'num_variants' => 'required|integer|min:1|max:5',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Create the ad
        $ad = Ad::create([
            'campaign_id' => $request->campaign_id,
            'brand_id' => $brand->id,
            'name' => $request->name,
            'product_info' => $request->product_info,
            'audience' => $request->audience,
            'tone' => $request->tone,
            'budget' => $request->budget,
            'currency' => $request->currency ?? 'USD',
            'status' => 'draft',
        ]);

        // Generate ad variants using AI service
        $this->generateAdVariants($ad, $request->num_variants);

        return redirect()->route('brand.ads.index')
            ->with('success', 'Ad created successfully! AI-generated variants are being processed.');
    }

    /**
     * Display the specified ad.
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function show(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $ad->load(['variants.performanceMetrics']);

        return view('brand.ads.show', compact('ad'));
    }

    /**
     * Show the form for editing the specified ad.
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function edit(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $brand = Auth::user()->brand;
        $campaigns = Campaign::where('brand_id', $brand->id)
            ->orderBy('name')
            ->get();

        return view('brand.ads.edit', compact('ad', 'campaigns'));
    }

    /**
     * Update the specified ad.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $validator = Validator::make($request->all(), [
            'campaign_id' => 'required|exists:campaigns,id,brand_id,' . Auth::user()->brand->id,
            'name' => 'required|string|max:255',
            'product_info' => 'required|string',
            'audience' => 'required|string',
            'tone' => 'required|string',
            'budget' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $ad->update([
            'campaign_id' => $request->campaign_id,
            'name' => $request->name,
            'product_info' => $request->product_info,
            'audience' => $request->audience,
            'tone' => $request->tone,
            'budget' => $request->budget,
            'currency' => $request->currency ?? 'USD',
        ]);

        return redirect()->route('brand.ads.show', $ad)
            ->with('success', 'Ad updated successfully!');
    }

    /**
     * Remove the specified ad.
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function destroy(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $ad->delete();

        return redirect()->route('brand.ads.index')
            ->with('success', 'Ad deleted successfully!');
    }

    /**
     * Launch selected ad variants.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function launchVariants(Request $request, Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $validator = Validator::make($request->all(), [
            'variant_ids' => 'required|array',
            'variant_ids.*' => 'exists:ad_variants,id,ad_id,' . $ad->id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Update selected variants to active
        AdVariant::whereIn('id', $request->variant_ids)
            ->where('ad_id', $ad->id)
            ->update(['is_active' => true]);

        // Update ad status to active
        $ad->update(['status' => 'active']);

        return redirect()->route('brand.ads.show', $ad)
            ->with('success', 'Ad variants launched successfully!');
    }

    /**
     * Pause selected ad variants.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function pauseVariants(Request $request, Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $validator = Validator::make($request->all(), [
            'variant_ids' => 'required|array',
            'variant_ids.*' => 'exists:ad_variants,id,ad_id,' . $ad->id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Update selected variants to inactive
        AdVariant::whereIn('id', $request->variant_ids)
            ->where('ad_id', $ad->id)
            ->update(['is_active' => false]);

        // Check if all variants are paused, then update ad status
        $activeVariantsCount = $ad->variants()->where('is_active', true)->count();
        if ($activeVariantsCount === 0) {
            $ad->update(['status' => 'paused']);
        }

        return redirect()->route('brand.ads.show', $ad)
            ->with('success', 'Ad variants paused successfully!');
    }

    /**
     * Duplicate an ad variant.
     *
     * @param  \App\Models\AdVariant  $variant
     * @return \Illuminate\Http\Response
     */
    public function duplicateVariant(AdVariant $variant)
    {
        // Ensure the variant belongs to the current brand
        if ($variant->ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        // Create a duplicate of the variant
        $newVariant = $variant->replicate();
        $newVariant->created_at = now();
        $newVariant->updated_at = now();
        $newVariant->is_active = false;
        $newVariant->is_winner = false;
        $newVariant->save();

        return redirect()->route('brand.ads.show', $variant->ad)
            ->with('success', 'Ad variant duplicated successfully!');
    }

    /**
     * Promote a variant as the winner.
     *
     * @param  \App\Models\AdVariant  $variant
     * @return \Illuminate\Http\Response
     */
    public function promoteWinner(AdVariant $variant)
    {
        // Ensure the variant belongs to the current brand
        if ($variant->ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        // Remove winner status from other variants of the same ad
        AdVariant::where('ad_id', $variant->ad_id)
            ->where('id', '!=', $variant->id)
            ->update(['is_winner' => false]);

        // Set this variant as the winner
        $variant->update(['is_winner' => true]);

        return redirect()->route('brand.ads.show', $variant->ad)
            ->with('success', 'Ad variant promoted as winner successfully!');
    }

    /**
     * Display A/B test dashboard for an ad.
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function abTestDashboard(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $ad->load(['variants.performanceMetrics']);

        // Calculate aggregate metrics for each variant
        $variantsWithMetrics = $ad->variants->map(function ($variant) {
            $metrics = $variant->performanceMetrics;
            
            return [
                'variant' => $variant,
                'total_impressions' => $metrics->sum('impressions'),
                'total_clicks' => $metrics->sum('clicks'),
                'total_conversions' => $metrics->sum('conversions'),
                'total_spend' => $metrics->sum('spend'),
                'avg_ctr' => $metrics->avg('ctr') ?? 0,
                'avg_conversion_rate' => $metrics->avg('conversion_rate') ?? 0,
                'avg_cpc' => $metrics->avg('cpc') ?? 0,
                'avg_roas' => $metrics->avg('roas') ?? 0,
            ];
        });

        // Determine the best performing variant
        $bestVariant = $variantsWithMetrics->sortByDesc('avg_ctr')->first();

        return view('brand.ads.ab-test-dashboard', compact('ad', 'variantsWithMetrics', 'bestVariant'));
    }

    /**
     * Export A/B test results.
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function exportResults(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $ad->load(['variants.performanceMetrics']);

        $csvData = [];
        $csvData[] = ['Variant ID', 'Headline', 'Body', 'CTA', 'Platform', 'Impressions', 'Clicks', 'Conversions', 'Spend', 'CTR', 'Conversion Rate', 'CPC', 'ROAS'];

        foreach ($ad->variants as $variant) {
            foreach ($variant->performanceMetrics as $metric) {
                $csvData[] = [
                    $variant->id,
                    $variant->headline,
                    $variant->body,
                    $variant->cta,
                    $metric->platform,
                    $metric->impressions,
                    $metric->clicks,
                    $metric->conversions,
                    $metric->spend,
                    $metric->ctr,
                    $metric->conversion_rate,
                    $metric->cpc,
                    $metric->roas,
                ];
            }
        }

        $filename = "ad_{$ad->id}_results_" . now()->format('Y-m-d_His') . ".csv";

        $callback = function() use ($csvData) {
            $file = fopen('php://output', 'w');
            foreach ($csvData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ]);
    }

    /**
     * Get real-time metrics for an ad (API endpoint).
     *
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMetrics(Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $variants = $ad->variants()->with('performanceMetrics')->get();

        $metricsData = $variants->map(function ($variant) {
            $metrics = $variant->performanceMetrics;
            
            return [
                'variant_id' => $variant->id,
                'headline' => $variant->headline,
                'is_active' => $variant->is_active,
                'is_winner' => $variant->is_winner,
                'metrics' => [
                    'impressions' => $metrics->sum('impressions'),
                    'clicks' => $metrics->sum('clicks'),
                    'conversions' => $metrics->sum('conversions'),
                    'spend' => $metrics->sum('spend'),
                    'ctr' => round($metrics->avg('ctr') ?? 0, 2),
                    'conversion_rate' => round($metrics->avg('conversion_rate') ?? 0, 2),
                    'cpc' => round($metrics->avg('cpc') ?? 0, 2),
                    'roas' => round($metrics->avg('roas') ?? 0, 2),
                ],
                'by_platform' => $metrics->groupBy('platform')->map(function ($platformMetrics) {
                    return [
                        'impressions' => $platformMetrics->sum('impressions'),
                        'clicks' => $platformMetrics->sum('clicks'),
                        'conversions' => $platformMetrics->sum('conversions'),
                        'ctr' => round($platformMetrics->avg('ctr') ?? 0, 2),
                    ];
                }),
            ];
        });

        return response()->json([
            'success' => true,
            'ad_id' => $ad->id,
            'ad_name' => $ad->name,
            'status' => $ad->status,
            'variants' => $metricsData,
        ]);
    }

    /**
     * Regenerate variants for an existing ad.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Ad  $ad
     * @return \Illuminate\Http\Response
     */
    public function regenerateVariants(Request $request, Ad $ad)
    {
        // Ensure the ad belongs to the current brand
        if ($ad->brand_id !== Auth::user()->brand->id) {
            abort(403);
        }

        $validator = Validator::make($request->all(), [
            'num_variants' => 'required|integer|min:1|max:5',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Generate new variants
        $this->generateAdVariants($ad, $request->num_variants);

        return redirect()->route('brand.ads.show', $ad)
            ->with('success', 'New ad variants generated successfully!');
    }

    /**
     * Generate ad variants using AI service.
     *
     * @param  \App\Models\Ad  $ad
     * @param  int  $numVariants
     * @return void
     */
    protected function generateAdVariants(Ad $ad, int $numVariants)
    {
        $adCopyGenerator = new AdCopyGeneratorService();
        
        try {
            $variants = $adCopyGenerator->generateVariants($ad, $numVariants);
            
            foreach ($variants as $variantData) {
                AdVariant::create([
                    'ad_id' => $ad->id,
                    'headline' => $variantData['headline'],
                    'body' => $variantData['body'],
                    'cta' => $variantData['cta'],
                    'predicted_metrics' => $variantData['predicted_metrics'],
                    'is_active' => false,
                    'is_winner' => false,
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('Failed to generate ad variants', [
                'ad_id' => $ad->id,
                'error' => $e->getMessage()
            ]);
            
            // Fallback to simple generation if AI service fails
            for ($i = 1; $i <= $numVariants; $i++) {
                AdVariant::create([
                    'ad_id' => $ad->id,
                    'headline' => "Generated Headline {$i} for {$ad->name}",
                    'body' => "This is a generated body text for variant {$i}. Product info: {$ad->product_info}. Audience: {$ad->audience}. Tone: {$ad->tone}.",
                    'cta' => "Learn More",
                    'predicted_metrics' => [
                        'predicted_ctr' => rand(30, 80) / 100,
                        'predicted_conversion_rate' => rand(1, 5) / 100,
                        'confidence_score' => 0.5,
                    ],
                    'is_active' => false,
                    'is_winner' => false,
                ]);
            }
        }
    }
}