<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\SocialAggregate;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;

class CreatorController extends Controller
{
    /**
     * Display a listing of creators with filtering options.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // Get influencer types from request
        $types = $request->input('types', []);
        
        // Build query for creators
        $query = User::whereHas('roles', function ($q) {
            $q->where('name', 'creator');
        })->with(['socialAggregates' => function ($q) {
            $q->whereIn('key', ['total_followers', 'total_posts']);
        }]);
        
        // Apply influencer type filtering
        if (!empty($types)) {
            $query->where(function ($q) use ($types) {
                // Check if "Other" is selected
                if (in_array('Other', $types)) {
                    // Include creators with "Other" type or custom type
                    $q->where(function ($subQ) {
                        $subQ->where('influencer_type', 'Other')
                             ->orWhereNotNull('influencer_type_custom')
                             ->where('influencer_type_custom', '!=', '')
                             ->where('influencer_type_custom', '!=', null);
                    });
                    
                    // Remove "Other" from types array for the next condition
                    $filteredTypes = array_diff($types, ['Other']);
                    
                    // Apply other selected types if any remain
                    if (!empty($filteredTypes)) {
                        $q->orWhereIn('influencer_type', $filteredTypes);
                    }
                } else {
                    // Apply selected types (excluding "Other")
                    $q->whereIn('influencer_type', $types);
                }
            });
        }
        
        // Paginate results
        $creators = $query->paginate(20)->appends(['types' => $types]);
        
        // Get all influencer types for filter dropdown
        $allTypes = [
            'Lifestyle', 'Tech', 'Gaming', 'Beauty', 'Fitness', 
            'Travel', 'Food', 'Education', 'Business', 
            'Entertainment', 'Sports', 'Other'
        ];
        
        return view('brand.creators.index', compact('creators', 'allTypes', 'types'));
    }
    
    /**
     * Display the specified creator.
     *
     * @param  \App\Models\User  $creator
     * @return \Illuminate\Http\Response
     */
    public function show(User $creator)
    {
        // Ensure the user is a creator
        if (!$creator->hasRole('creator')) {
            abort(404);
        }
        
        // Load social aggregates for the creator
        $creator->load(['socialAggregates' => function ($q) {
            $q->whereIn('key', ['total_followers', 'total_posts', 'total_views', 'engagement_rate']);
        }]);
        
        // Get connection status
        $connections = $creator->socialAccounts()->orderBy('provider')->get();
        
        // Get recent activity
        $recentActivity = \App\Models\SocialPost::whereHas('socialAccount', function ($query) use ($creator) {
                $query->where('user_id', $creator->id);
            })
            ->orderBy('published_at', 'desc')
            ->limit(10)
            ->get();
        
        return view('brand.creators.show', compact('creator', 'connections', 'recentActivity'));
    }
    
    /**
     * Show the form for creating a proposal to the specified creator.
     *
     * @param  \App\Models\User  $creator
     * @return \Illuminate\Http\Response
     */
    public function createProposal(User $creator)
    {
        // Ensure the user is a creator
        if (!$creator->hasRole('creator')) {
            abort(404);
        }
        
        return view('brand.creators.proposal.create', compact('creator'));
    }
    
    /**
     * Store a newly created proposal for the specified creator.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\User  $creator
     * @return \Illuminate\Http\Response
     */
    public function storeProposal(Request $request, User $creator)
    {
        // Ensure the user is a creator
        if (!$creator->hasRole('creator')) {
            abort(404);
        }
        
        // Validate the request
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:Collaboration,Sponsored Post,Affiliate,Long-term Partnership',
            'budget' => 'nullable|numeric|min:0',
            'duration' => 'nullable|string|max:255',
            'deliverables' => 'nullable|array',
            'deliverables.*' => 'string',
            'attachment' => 'nullable|file|mimes:pdf|max:2048', // 2MB max
        ]);
        
        // Handle file upload
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('proposals', 'public');
        }
        
        // Create the proposal
        $proposal = new \App\Models\Proposal([
            'brand_id' => auth()->user()->brand_id,
            'creator_id' => $creator->id,
            'title' => $validated['title'],
            'message' => $validated['message'],
            'type' => $validated['type'],
            'budget' => $validated['budget'] ?? null,
            'duration' => $validated['duration'] ?? null,
            'deliverables' => $validated['deliverables'] ?? null,
            'attachment_path' => $attachmentPath,
            'status' => 'sent',
        ]);
        
        $proposal->save();
        
        // Send notification to creator
        $creator->notify(new \App\Notifications\ProposalReceived($proposal));
        
        // Redirect back to creator profile with success message
        return redirect()->route('brand.creators.show', $creator)->with('success', 'Proposal sent successfully!');
    }
    
    /**
     * Return creators data as JSON for AJAX filtering.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function data(Request $request)
    {
        // Get influencer types from request
        $types = $request->input('types', []);
        
        // Build query for creators
        $query = User::whereHas('roles', function ($q) {
            $q->where('name', 'creator');
        })->with(['socialAggregates' => function ($q) {
            $q->whereIn('key', ['total_followers', 'total_posts']);
        }]);
        
        // Apply influencer type filtering
        if (!empty($types)) {
            $query->where(function ($q) use ($types) {
                // Check if "Other" is selected
                if (in_array('Other', $types)) {
                    // Include creators with "Other" type or custom type
                    $q->where(function ($subQ) {
                        $subQ->where('influencer_type', 'Other')
                             ->orWhereNotNull('influencer_type_custom')
                             ->where('influencer_type_custom', '!=', '')
                             ->where('influencer_type_custom', '!=', null);
                    });
                    
                    // Remove "Other" from types array for the next condition
                    $filteredTypes = array_diff($types, ['Other']);
                    
                    // Apply other selected types if any remain
                    if (!empty($filteredTypes)) {
                        $q->orWhereIn('influencer_type', $filteredTypes);
                    }
                } else {
                    // Apply selected types (excluding "Other")
                    $q->whereIn('influencer_type', $types);
                }
            });
        }
        
        // Paginate results
        $creators = $query->paginate(20);
        
        // Transform data for JSON response
        $data = $creators->map(function ($creator) {
            $followers = $creator->socialAggregates->firstWhere('key', 'total_followers');
            $posts = $creator->socialAggregates->firstWhere('key', 'total_posts');
            
            return [
                'id' => $creator->id,
                'name' => $creator->name,
                'email' => $creator->email,
                'avatar_url' => $creator->avatar_url,
                'influencer_type' => $creator->influencer_type === 'Other' && $creator->influencer_type_custom 
                    ? $creator->influencer_type_custom 
                    : $creator->influencer_type,
                'total_followers' => $followers ? $followers->getNumericValue() : 0,
                'total_posts' => $posts ? $posts->getNumericValue() : 0,
                'last_active_at' => $creator->updated_at->format('M j, Y H:i'),
            ];
        });
        
        return response()->json([
            'data' => $data,
            'total' => $creators->total(),
            'current_page' => $creators->currentPage(),
            'last_page' => $creators->lastPage(),
            'per_page' => $creators->perPage(),
        ]);
    }
}