<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\LocalizedContent;
use App\Models\Brand;
use App\Services\LocalizationService;
use App\Services\AITranslationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\Rule;
use Exception;

class LocalizationController extends Controller
{
    protected LocalizationService $localizationService;

    public function __construct(LocalizationService $localizationService)
    {
        $this->localizationService = $localizationService;
    }

    /**
     * Display localization panel for campaign manager.
     */
    public function index(Request $request)
    {
        $brand = Brand::findOrFail($request->input('brand_id'));
        
        // Check authorization
        Gate::authorize('manage-localization', $brand);

        $contentType = $request->input('content_type');
        $contentId = $request->input('content_id');

        $localizations = [];
        if ($contentType && $contentId) {
            $localizations = $this->localizationService->getContentLocalizations($contentType, $contentId);
        }

        $stats = $this->localizationService->getBrandLocalizationStats($brand->id);
        $supportedLocales = AITranslationService::getSupportedLocales();
        $supportedTones = AITranslationService::getSupportedTones();

        return view('brand.localization.index', compact(
            'brand',
            'localizations',
            'stats',
            'supportedLocales',
            'supportedTones',
            'contentType',
            'contentId'
        ));
    }

    /**
     * Request new localization.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'content_type' => 'required|string|in:ad,post,proposal,campaign',
            'content_id' => 'required|integer',
            'source_text' => 'required|string|max:10000',
            'target_locales' => 'required|array|min:1',
            'target_locales.*' => ['required', 'string', Rule::in(array_keys(AITranslationService::getSupportedLocales()))],
            'tone' => ['required', 'string', Rule::in(['formal', 'friendly', 'neutral'])],
            'context' => 'nullable|string|max:1000',
        ]);

        $brand = Brand::findOrFail($validated['brand_id']);
        
        // Check authorization
        Gate::authorize('manage-localization', $brand);

        try {
            $results = $this->localizationService->requestLocalization(
                $validated['brand_id'],
                $validated['content_type'],
                $validated['content_id'],
                $validated['source_text'],
                $validated['target_locales'],
                $validated['tone'],
                $request->user(),
                $validated['context'] ?? null
            );

            return response()->json([
                'success' => true,
                'message' => 'Localization request processed successfully',
                'data' => $results,
            ], 201);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to process localization request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Show specific localized content.
     */
    public function show(LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('view', $localizedContent);

        $localizedContent->load(['creator', 'approver', 'auditLogs.user']);

        return response()->json([
            'success' => true,
            'data' => $localizedContent,
        ]);
    }

    /**
     * Approve localized content.
     */
    public function approve(Request $request, LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('approve', $localizedContent);

        try {
            $updated = $this->localizationService->approveLocalization(
                $localizedContent,
                $request->user()
            );

            return response()->json([
                'success' => true,
                'message' => 'Localization approved successfully',
                'data' => $updated,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve localization',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Reject localized content.
     */
    public function reject(Request $request, LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('approve', $localizedContent);

        $validated = $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        try {
            $updated = $this->localizationService->rejectLocalization(
                $localizedContent,
                $request->user(),
                $validated['reason'] ?? null
            );

            return response()->json([
                'success' => true,
                'message' => 'Localization rejected',
                'data' => $updated,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reject localization',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Edit localized content manually.
     */
    public function update(Request $request, LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('update', $localizedContent);

        $validated = $request->validate([
            'localized_text' => 'required|string|max:10000',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $updated = $this->localizationService->editLocalization(
                $localizedContent,
                $validated['localized_text'],
                $request->user(),
                $validated['notes'] ?? null
            );

            return response()->json([
                'success' => true,
                'message' => 'Localization updated successfully',
                'data' => $updated,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update localization',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Export localized content.
     */
    public function export(LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('view', $localizedContent);

        try {
            $exported = $this->localizationService->exportLocalization(
                $localizedContent,
                request()->user()
            );

            return response()->json([
                'success' => true,
                'data' => $exported,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to export localization',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get all localizations for specific content.
     */
    public function getContentLocalizations(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'content_type' => 'required|string|in:ad,post,proposal,campaign',
            'content_id' => 'required|integer',
        ]);

        try {
            $localizations = $this->localizationService->getContentLocalizations(
                $validated['content_type'],
                $validated['content_id']
            );

            return response()->json([
                'success' => true,
                'data' => $localizations,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve localizations',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get localization statistics for brand.
     */
    public function stats(Brand $brand): JsonResponse
    {
        Gate::authorize('manage-localization', $brand);

        try {
            $stats = $this->localizationService->getBrandLocalizationStats($brand->id);

            return response()->json([
                'success' => true,
                'data' => $stats,
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve statistics',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get supported locales.
     */
    public function supportedLocales(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => AITranslationService::getSupportedLocales(),
        ]);
    }

    /**
     * Get supported tones.
     */
    public function supportedTones(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => AITranslationService::getSupportedTones(),
        ]);
    }

    /**
     * Delete localized content.
     */
    public function destroy(LocalizedContent $localizedContent): JsonResponse
    {
        Gate::authorize('delete', $localizedContent);

        try {
            $localizedContent->delete();

            return response()->json([
                'success' => true,
                'message' => 'Localization deleted successfully',
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete localization',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}