<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\ContentIdea;
use App\Models\SocialAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class ContentIdeaController extends Controller
{
    /**
     * Display a listing of content ideas.
     */
    public function index()
    {
        $user = Auth::user();
        
        $contentIdeas = ContentIdea::where('creator_id', $user->id)
            ->with(['user'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('creator.content-ideas.index', compact('contentIdeas'));
    }

    /**
     * Show the form for generating content ideas.
     */
    public function create()
    {
        $user = Auth::user();
        
        // Get creator's social accounts
        $socialAccounts = SocialAccount::where('user_id', $user->id)
            ->orderBy('provider')
            ->get();
            
        $platforms = $socialAccounts->pluck('provider')->unique()->toArray();

        return view('creator.content-ideas.create', compact('platforms'));
    }

    /**
     * Store a newly created content idea request.
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        
        // Validate request
        $request->validate([
            'topic' => 'required|string|max:255',
            'platform' => 'required|string|max:50',
            'style' => 'required|string|max:50',
            'target_audience' => 'required|string|max:255',
            'number_of_ideas' => 'required|integer|min:1|max:20',
        ]);

        // Create AI job for content idea generation
        $aiJob = \App\Models\AiJob::create([
            'job_id' => (string) Str::uuid(),
            'type' => 'content_idea_generation',
            'status' => 'queued',
            'brand_id' => null,
            'creator_id' => $user->id,
            'proposal_id' => null,
            'result_text' => null,
            'result_meta' => null,
            'provider' => null,
            'error_message' => null,
            'token_usage' => 0,
        ]);

        // Prepare payload for AI processing
        $payload = [
            'topic' => $request->topic,
            'platform' => $request->platform,
            'style' => $request->style,
            'target_audience' => $request->target_audience,
            'number_of_ideas' => $request->number_of_ideas,
            'creator_name' => $user->name,
            'creator_niche' => $user->influencer_type ?? 'general',
        ];

        // Dispatch job to process the content idea generation
        \App\Jobs\GenerateCreatorContentIdeasJob::dispatch($aiJob->id, $payload);

        return redirect()->route('creator.content-ideas.index')
            ->with('success', 'Content ideas generation started! Check back soon for results.');
    }

    /**
     * Save a content idea as favorite.
     */
    public function favorite($id)
    {
        $user = Auth::user();
        
        $contentIdea = ContentIdea::findOrFail($id);

        // Check if content idea belongs to the creator
        if ($contentIdea->creator_id !== $user->id) {
            return redirect()->back()->with('error', 'Unauthorized action.');
        }

        // Update status to favorite
        $contentIdea->update(['status' => 'favorite']);

        return redirect()->back()->with('success', 'Content idea marked as favorite!');
    }

    /**
     * Copy a content idea.
     */
    public function copy($id)
    {
        $user = Auth::user();
        
        $contentIdea = ContentIdea::findOrFail($id);

        // Check if content idea belongs to the creator
        if ($contentIdea->creator_id !== $user->id) {
            return redirect()->back()->with('error', 'Unauthorized action.');
        }

        // Return the content idea data as JSON for copying
        return response()->json([
            'caption' => $contentIdea->caption,
            'hashtags' => $contentIdea->hashtags,
            'creative_hints' => $contentIdea->creative_hints,
        ]);
    }

    /**
     * Save a content idea to draft.
     */
    public function saveToDraft($id)
    {
        $user = Auth::user();
        
        $contentIdea = ContentIdea::findOrFail($id);

        // Check if content idea belongs to the creator
        if ($contentIdea->creator_id !== $user->id) {
            return redirect()->back()->with('error', 'Unauthorized action.');
        }

        // Update status to draft
        $contentIdea->update(['status' => 'draft']);

        return redirect()->back()->with('success', 'Content idea saved to draft!');
    }

    /**
     * Export a content idea to a proposal.
     */
    public function exportToProposal($id)
    {
        $user = Auth::user();
        
        $contentIdea = ContentIdea::findOrFail($id);

        // Check if content idea belongs to the creator
        if ($contentIdea->creator_id !== $user->id) {
            return redirect()->back()->with('error', 'Unauthorized action.');
        }

        // In a real implementation, this would create a proposal
        // For now, we'll just show a success message
        return redirect()->back()->with('success', 'Content idea exported to proposal successfully!');
    }
}