<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SidebarController extends Controller
{
    /**
     * Get badge counts for creator sidebar navigation
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBadges(Request $request)
    {
        $user = $request->user();
        $creatorId = $user->id;

        // Get pending proposals count (received from brands)
        $pendingProposals = DB::table('proposals')
            ->where('creator_id', $creatorId)
            ->whereIn('status', ['sent', 'negotiation'])
            ->count();

        // Get active campaigns count (campaigns creator is participating in)
        $activeCampaigns = DB::table('campaign_creators')
            ->join('campaigns', 'campaign_creators.campaign_id', '=', 'campaigns.id')
            ->where('campaign_creators.creator_id', $creatorId)
            ->where('campaigns.status', 'active')
            ->count();

        // Get unread messages count
        $unreadMessages = DB::table('message_threads')
            ->join('message_thread_participants', 'message_threads.id', '=', 'message_thread_participants.thread_id')
            ->join('messages', 'message_threads.id', '=', 'messages.thread_id')
            ->where('message_thread_participants.user_id', $creatorId)
            ->where('messages.sender_id', '!=', $creatorId)
            ->whereNull('messages.read_at')
            ->distinct('message_threads.id')
            ->count('message_threads.id');

        // Get notifications count (unread)
        $notifications = $user->unreadNotifications()->count();

        // Get pending payments count
        $pendingPayments = DB::table('payments')
            ->where('creator_id', $creatorId)
            ->whereIn('status', ['pending', 'processing'])
            ->count();

        return response()->json([
            'success' => true,
            'badges' => [
                'pendingProposals' => $pendingProposals,
                'activeCampaigns' => $activeCampaigns,
                'unreadMessages' => $unreadMessages,
                'notifications' => $notifications,
                'pendingPayments' => $pendingPayments,
            ]
        ]);
    }

    /**
     * Get quick search results for creator
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function quickSearch(Request $request)
    {
        $query = $request->input('query', '');
        $user = $request->user();
        $creatorId = $user->id;

        if (strlen($query) < 2) {
            return response()->json([
                'success' => true,
                'results' => []
            ]);
        }

        $results = [];

        // Search campaigns (that creator is part of)
        $campaigns = DB::table('campaigns')
            ->join('campaign_creators', 'campaigns.id', '=', 'campaign_creators.campaign_id')
            ->where('campaign_creators.creator_id', $creatorId)
            ->where('campaigns.name', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['campaigns.id', 'campaigns.name', 'campaigns.status']);

        foreach ($campaigns as $campaign) {
            $results[] = [
                'name' => $campaign->name,
                'category' => 'Campaign',
                'url' => route('creator.campaigns.show', $campaign->id),
                'meta' => "Status: {$campaign->status}"
            ];
        }

        // Search content ideas
        $contentIdeas = DB::table('content_ideas')
            ->where('creator_id', $creatorId)
            ->where(function($q) use ($query) {
                $q->where('title', 'LIKE', "%{$query}%")
                  ->orWhere('description', 'LIKE', "%{$query}%");
            })
            ->limit(5)
            ->get(['id', 'title', 'status']);

        foreach ($contentIdeas as $idea) {
            $results[] = [
                'name' => $idea->title,
                'category' => 'Content Idea',
                'url' => route('creator.content-ideas.show', $idea->id),
                'meta' => "Status: {$idea->status}"
            ];
        }

        // Search proposals (received)
        $proposals = DB::table('proposals')
            ->where('creator_id', $creatorId)
            ->where('title', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['id', 'title', 'status']);

        foreach ($proposals as $proposal) {
            $results[] = [
                'name' => $proposal->title,
                'category' => 'Proposal',
                'url' => route('creator.proposals.show', $proposal->id),
                'meta' => "Status: {$proposal->status}"
            ];
        }

        // Search scheduled posts
        $scheduledPosts = DB::table('creator_scheduled_posts')
            ->where('creator_id', $creatorId)
            ->where('caption', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['id', 'caption', 'status', 'scheduled_at']);

        foreach ($scheduledPosts as $post) {
            $caption = strlen($post->caption) > 50 ? substr($post->caption, 0, 50) . '...' : $post->caption;
            $results[] = [
                'name' => $caption,
                'category' => 'Scheduled Post',
                'url' => route('creator.scheduled-posts.show', $post->id),
                'meta' => "Scheduled: " . date('M d, Y', strtotime($post->scheduled_at))
            ];
        }

        return response()->json([
            'success' => true,
            'results' => $results
        ]);
    }

    /**
     * Get creator dashboard stats
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDashboardStats(Request $request)
    {
        $user = $request->user();
        $creatorId = $user->id;

        // Get total followers across all platforms
        $totalFollowers = DB::table('social_accounts')
            ->where('user_id', $creatorId)
            ->sum('followers_count');

        // Get average engagement rate
        $avgEngagement = DB::table('social_accounts')
            ->where('user_id', $creatorId)
            ->avg('engagement_rate');

        // Get total earnings
        $totalEarnings = DB::table('payments')
            ->where('creator_id', $creatorId)
            ->where('status', 'completed')
            ->sum('amount');

        // Get pending earnings
        $pendingEarnings = DB::table('payments')
            ->where('creator_id', $creatorId)
            ->whereIn('status', ['pending', 'processing'])
            ->sum('amount');

        // Get active campaigns count
        $activeCampaigns = DB::table('campaign_creators')
            ->join('campaigns', 'campaign_creators.campaign_id', '=', 'campaigns.id')
            ->where('campaign_creators.creator_id', $creatorId)
            ->where('campaigns.status', 'active')
            ->count();

        // Get campaign invites count
        $campaignInvites = DB::table('proposals')
            ->where('creator_id', $creatorId)
            ->where('status', 'sent')
            ->count();

        return response()->json([
            'success' => true,
            'stats' => [
                'totalFollowers' => $totalFollowers,
                'avgEngagement' => round($avgEngagement, 2),
                'totalEarnings' => $totalEarnings,
                'pendingEarnings' => $pendingEarnings,
                'activeCampaigns' => $activeCampaigns,
                'campaignInvites' => $campaignInvites,
            ]
        ]);
    }

    /**
     * Mark notification as read
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function markNotificationRead(Request $request)
    {
        $request->validate([
            'notification_id' => 'required|string'
        ]);

        $user = $request->user();
        $notification = $user->notifications()->find($request->notification_id);

        if ($notification) {
            $notification->markAsRead();
            
            return response()->json([
                'success' => true,
                'message' => 'Notification marked as read'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Notification not found'
        ], 404);
    }

    /**
     * Mark all notifications as read
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function markAllNotificationsRead(Request $request)
    {
        $user = $request->user();
        $user->unreadNotifications->markAsRead();

        return response()->json([
            'success' => true,
            'message' => 'All notifications marked as read'
        ]);
    }
}