<?php

namespace App\Http\Controllers\Creator;

use App\Http\Controllers\Controller;
use App\Models\CreatorSocialConnection;
use App\Models\SharedSocialAccess;
use App\Models\Brand;
use App\Services\SocialConnectionService;
use App\Services\OAuthService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\Rule;

class SocialConnectionController extends Controller
{
    protected $connectionService;
    protected $oauthService;

    public function __construct(SocialConnectionService $connectionService, OAuthService $oauthService)
    {
        $this->connectionService = $connectionService;
        $this->oauthService = $oauthService;
    }

    /**
     * Display a listing of creator's social connections.
     */
    public function index(Request $request)
    {
        $creator = $request->user();

        $connections = CreatorSocialConnection::where('creator_id', $creator->id)
            ->with(['sharedAccess.brand', 'sharedAccess.campaign'])
            ->orderBy('created_at', 'desc')
            ->get();

        $platforms = $this->getAvailablePlatforms();

        return view('creator.social-connections.index', compact('connections', 'platforms'));
    }

    /**
     * Show the form for creating a new connection.
     */
    public function create(Request $request)
    {
        $platforms = $this->getAvailablePlatforms();
        $platform = $request->query('platform');

        return view('creator.social-connections.create', compact('platforms', 'platform'));
    }

    /**
     * Store a newly created connection.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'platform' => ['required', 'string', Rule::in(['instagram', 'youtube', 'twitter', 'tiktok', 'linkedin', 'facebook'])],
            'label' => 'nullable|string|max:255',
            'connection_type' => ['required', 'string', Rule::in(['oauth', 'manual'])],
            'access_token' => 'required_if:connection_type,manual|nullable|string',
            'refresh_token' => 'nullable|string',
            'client_id' => 'nullable|string',
            'client_secret' => 'nullable|string',
            'scopes' => 'nullable|array',
        ]);

        $creator = $request->user();

        // Check if connection already exists for this platform
        $existingConnection = CreatorSocialConnection::where('creator_id', $creator->id)
            ->where('platform', $validated['platform'])
            ->first();

        if ($existingConnection) {
            return back()->withErrors(['platform' => 'You already have a connection for this platform. Please disconnect it first.']);
        }

        // Create connection
        $connection = CreatorSocialConnection::create([
            'creator_id' => $creator->id,
            'platform' => $validated['platform'],
            'label' => $validated['label'],
            'access_token' => $validated['access_token'] ?? null,
            'refresh_token' => $validated['refresh_token'] ?? null,
            'client_id' => $validated['client_id'] ?? null,
            'client_secret' => $validated['client_secret'] ?? null,
            'scopes' => $validated['scopes'] ?? [],
            'status' => 'disconnected',
            'created_by' => $creator->id,
            'updated_by' => $creator->id,
        ]);

        // Test connection if manual token provided
        if ($validated['connection_type'] === 'manual' && $validated['access_token']) {
            $result = $this->connectionService->testConnection($connection);
            
            if (!$result['success']) {
                return back()->with('warning', 'Connection created but test failed: ' . $result['message']);
            }
        }

        return redirect()->route('creator.social-connections.index')
            ->with('success', 'Social connection added successfully!');
    }

    /**
     * Display the specified connection.
     */
    public function show(CreatorSocialConnection $connection)
    {
        Gate::authorize('view', $connection);

        $connection->load(['sharedAccess.brand', 'sharedAccess.campaign']);

        return view('creator.social-connections.show', compact('connection'));
    }

    /**
     * Update the specified connection.
     */
    public function update(Request $request, CreatorSocialConnection $connection)
    {
        Gate::authorize('update', $connection);

        $validated = $request->validate([
            'label' => 'nullable|string|max:255',
            'access_token' => 'nullable|string',
            'refresh_token' => 'nullable|string',
            'use_for_ads' => 'nullable|boolean',
        ]);

        $validated['updated_by'] = $request->user()->id;

        $connection->update($validated);

        return back()->with('success', 'Connection updated successfully!');
    }

    /**
     * Remove the specified connection.
     */
    public function destroy(CreatorSocialConnection $connection)
    {
        Gate::authorize('delete', $connection);

        // Revoke all active shared access
        $connection->activeSharedAccess()->each(function ($sharedAccess) {
            $this->connectionService->revokeSharedAccess($sharedAccess, 'Connection deleted by creator');
        });

        $connection->delete();

        return redirect()->route('creator.social-connections.index')
            ->with('success', 'Connection deleted successfully!');
    }

    /**
     * Test connection.
     */
    public function test(CreatorSocialConnection $connection)
    {
        Gate::authorize('update', $connection);

        $result = $this->connectionService->testConnection($connection);

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Connection test successful!',
                'data' => $result['data'],
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'],
        ], 400);
    }

    /**
     * Refresh OAuth token.
     */
    public function refreshToken(CreatorSocialConnection $connection)
    {
        Gate::authorize('update', $connection);

        $success = $this->connectionService->refreshToken($connection);

        if ($success) {
            return back()->with('success', 'Token refreshed successfully!');
        }

        return back()->with('error', 'Failed to refresh token. Please reconnect your account.');
    }

    /**
     * Initiate OAuth flow.
     */
    public function oauth(Request $request, string $platform)
    {
        $validated = $request->validate([
            'label' => 'nullable|string|max:255',
            'scopes' => 'nullable|array',
        ]);

        $creator = $request->user();

        // Store connection details in session for callback
        session([
            'oauth_connection_data' => [
                'platform' => $platform,
                'label' => $validated['label'] ?? null,
                'scopes' => $validated['scopes'] ?? [],
            ]
        ]);

        // Generate OAuth URL (using existing OAuthService but for creator context)
        $authUrl = $this->oauthService->getAuthorizationUrl(
            $platform,
            $creator->brand ?? (object)['id' => $creator->id], // Adapt for creator
            $validated['scopes'] ?? []
        );

        return redirect($authUrl);
    }

    /**
     * Handle OAuth callback.
     */
    public function oauthCallback(Request $request, string $platform)
    {
        $code = $request->query('code');
        $state = $request->query('state');

        if (!$code || !$state) {
            return redirect()->route('creator.social-connections.index')
                ->with('error', 'OAuth authorization failed or was cancelled.');
        }

        try {
            // Exchange code for tokens
            $tokens = $this->oauthService->handleCallback($platform, $code, $state);

            // Get connection data from session
            $connectionData = session('oauth_connection_data');
            session()->forget('oauth_connection_data');

            $creator = $request->user();

            // Create connection
            $connection = CreatorSocialConnection::create([
                'creator_id' => $creator->id,
                'platform' => $platform,
                'label' => $connectionData['label'] ?? null,
                'access_token' => $tokens['access_token'],
                'refresh_token' => $tokens['refresh_token'] ?? null,
                'scopes' => $connectionData['scopes'] ?? [],
                'token_expires_at' => now()->addSeconds($tokens['expires_in'] ?? 3600),
                'status' => 'connected',
                'created_by' => $creator->id,
                'updated_by' => $creator->id,
            ]);

            // Test connection to get profile info
            $this->connectionService->testConnection($connection);

            return redirect()->route('creator.social-connections.index')
                ->with('success', 'Social account connected successfully!');

        } catch (\Exception $e) {
            return redirect()->route('creator.social-connections.index')
                ->with('error', 'Failed to connect account: ' . $e->getMessage());
        }
    }

    /**
     * Share connection with a brand.
     */
    public function share(Request $request, CreatorSocialConnection $connection)
    {
        Gate::authorize('update', $connection);

        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'scopes' => 'required|array|min:1',
            'scopes.*' => ['required', 'string', Rule::in(['read_profile', 'read_posts', 'read_analytics', 'publish_content', 'manage_comments', 'manage_ads'])],
            'access_level' => ['required', 'string', Rule::in(['read_only', 'read_write', 'full'])],
            'expiry_days' => 'required|integer|min:1|max:365',
        ]);

        $brand = Brand::findOrFail($validated['brand_id']);

        // Check if already shared
        if ($connection->isSharedWith($brand->id)) {
            return back()->withErrors(['brand_id' => 'This connection is already shared with this brand.']);
        }

        try {
            $sharedAccess = $this->connectionService->shareWithBrand(
                $connection,
                $brand,
                $validated['scopes'],
                $validated['access_level'],
                $validated['expiry_days'],
                $validated['campaign_id'] ?? null
            );

            return back()->with('success', 'Connection shared with brand successfully!');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to share connection: ' . $e->getMessage());
        }
    }

    /**
     * Revoke shared access.
     */
    public function revokeAccess(Request $request, SharedSocialAccess $sharedAccess)
    {
        Gate::authorize('revoke', $sharedAccess);

        $validated = $request->validate([
            'reason' => 'nullable|string|max:500',
        ]);

        $this->connectionService->revokeSharedAccess($sharedAccess, $validated['reason'] ?? null);

        return back()->with('success', 'Access revoked successfully!');
    }

    /**
     * Get available platforms with metadata.
     */
    protected function getAvailablePlatforms(): array
    {
        return [
            [
                'slug' => 'instagram',
                'name' => 'Instagram',
                'icon' => 'fab fa-instagram',
                'color' => '#E4405F',
                'supports_oauth' => true,
                'supports_manual' => true,
                'required_scopes' => ['instagram_basic', 'instagram_content_publish'],
                'description' => 'Connect your Instagram account to share posts and analytics.',
            ],
            [
                'slug' => 'youtube',
                'name' => 'YouTube',
                'icon' => 'fab fa-youtube',
                'color' => '#FF0000',
                'supports_oauth' => true,
                'supports_manual' => false,
                'required_scopes' => ['youtube.readonly', 'youtube.upload'],
                'description' => 'Connect your YouTube channel to share videos and analytics.',
            ],
            [
                'slug' => 'twitter',
                'name' => 'X (Twitter)',
                'icon' => 'fab fa-x-twitter',
                'color' => '#000000',
                'supports_oauth' => true,
                'supports_manual' => true,
                'required_scopes' => ['tweet.read', 'users.read', 'offline.access'],
                'description' => 'Connect your X account to share tweets and engagement data.',
            ],
            [
                'slug' => 'tiktok',
                'name' => 'TikTok',
                'icon' => 'fab fa-tiktok',
                'color' => '#000000',
                'supports_oauth' => true,
                'supports_manual' => false,
                'required_scopes' => ['user.info.basic', 'video.list'],
                'description' => 'Connect your TikTok account to share videos and analytics.',
            ],
            [
                'slug' => 'linkedin',
                'name' => 'LinkedIn',
                'icon' => 'fab fa-linkedin',
                'color' => '#0077B5',
                'supports_oauth' => true,
                'supports_manual' => false,
                'required_scopes' => ['r_liteprofile', 'w_member_social'],
                'description' => 'Connect your LinkedIn profile to share posts and professional content.',
            ],
            [
                'slug' => 'facebook',
                'name' => 'Facebook',
                'icon' => 'fab fa-facebook',
                'color' => '#1877F2',
                'supports_oauth' => true,
                'supports_manual' => true,
                'required_scopes' => ['pages_show_list', 'pages_read_engagement'],
                'description' => 'Connect your Facebook page to share posts and insights.',
            ],
        ];
    }
}