<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use App\Services\AnalyticsService;
use Illuminate\Http\Request;

class MarketingController extends Controller
{
    protected $analytics;

    public function __construct(AnalyticsService $analytics)
    {
        $this->analytics = $analytics;
    }

    /**
     * Show marketing home page
     */
    public function home(Request $request)
    {
        $role = $request->get('role', 'brand');
        
        return view('marketing.home', [
            'selectedRole' => $role
        ]);
    }

    /**
     * Show pricing page
     */
    public function pricing(Request $request)
    {
        $role = $request->get('role', 'all');
        $interval = $request->get('interval', 'monthly');
        
        // Track pricing page view
        $this->analytics->trackPricingView($role, $interval);
        
        $plans = Plan::active()->ordered()->get();
        
        // Calculate yearly savings (2 months free)
        $savings = $plans->mapWithKeys(function ($plan) {
            return [$plan->id => $plan->price * 2];
        });
        
        return view('marketing.pricing', [
            'plans' => $plans,
            'selectedRole' => $role,
            'interval' => $interval,
            'savings' => $savings
        ]);
    }

    /**
     * Show features page
     */
    public function features()
    {
        return view('marketing.features');
    }

    /**
     * Show demo page
     */
    public function demo()
    {
        return view('marketing.demo');
    }

    /**
     * Show help center
     */
    public function help()
    {
        return view('marketing.help');
    }

    /**
     * Show brand onboarding
     */
    public function onboardingBrand()
    {
        return view('marketing.onboarding.brand');
    }

    /**
     * Show creator onboarding
     */
    public function onboardingCreator()
    {
        return view('marketing.onboarding.creator');
    }

    /**
     * Show agency onboarding
     */
    public function onboardingAgency()
    {
        return view('marketing.onboarding.agency');
    }

    /**
     * Contact sales (Enterprise)
     */
    public function contactSales(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'company' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'message' => 'required|string|max:1000'
        ]);

        // Track contact sales event
        $this->analytics->trackEvent('contact_sales', [
            'company' => $validated['company']
        ]);

        // Send email to sales team
        // \Mail::to(config('mail.sales_email'))->send(new \App\Mail\SalesInquiry($validated));

        return back()->with('success', 'Thank you! Our sales team will contact you within 24 hours.');
    }
}