<?php

namespace App\Jobs;

use App\Models\ScheduledPost;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class DispatchDueScheduledPosts implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            // Log start of dispatching
            Log::info('Starting to dispatch due scheduled posts');
            
            // Find all scheduled posts that are due
            $duePosts = ScheduledPost::where('status', 'scheduled')
                ->where('publish_time', '<=', now())
                ->get();
                
            Log::info('Found due scheduled posts', ['count' => $duePosts->count()]);

            foreach ($duePosts as $post) {
                Log::info('Dispatching scheduled post for processing', [
                    'scheduled_post_id' => $post->id,
                    'brand_id' => $post->brand_id,
                ]);
                
                // Dispatch the processing job
                ProcessScheduledPost::dispatch($post);
            }

            Log::info('Dispatched ' . $duePosts->count() . ' scheduled posts for processing');
        } catch (\Exception $e) {
            Log::error('Failed to dispatch due scheduled posts: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
