<?php

namespace App\Jobs;

use App\Models\AiJob;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class GenerateProposalJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $aiJobId;
    protected $payload;

    /**
     * Create a new job instance.
     *
     * @param int $aiJobId
     * @param array $payload
     * @return void
     */
    public function __construct($aiJobId, $payload)
    {
        $this->aiJobId = $aiJobId;
        $this->payload = $payload;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $aiJob = AiJob::find($this->aiJobId);
        
        if (!$aiJob) {
            Log::error("AI Job not found: " . $this->aiJobId);
            return;
        }
        
        // Update status to generating
        $aiJob->update(['status' => 'generating']);
        
        try {
            // Simulate AI processing delay
            sleep(3);
            
            // Generate proposal content based on payload
            $proposalContent = $this->generateProposalContent($this->payload);
            
            // Update AI job with results
            $aiJob->update([
                'status' => 'ready',
                'result_text' => $proposalContent['text'],
                'result_meta' => $proposalContent['meta'],
                'provider' => 'openai-gpt-4',
                'token_usage' => $proposalContent['token_usage'],
                'completed_at' => now(),
            ]);
            
            // Log token usage for cost tracking
            Log::info("AI Proposal generated for job: " . $aiJob->job_id . " with token usage: " . $proposalContent['token_usage']);
        } catch (\Exception $e) {
            // Update AI job with error
            $aiJob->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);
            
            Log::error("AI Proposal generation failed for job: " . $aiJob->job_id . " with error: " . $e->getMessage());
        }
    }
    
    /**
     * Generate proposal content based on payload.
     *
     * @param array $payload
     * @return array
     */
    protected function generateProposalContent($payload)
    {
        // In a real implementation, this would call an AI service
        // For now, we'll generate mock content
        
        $title = "Collaboration Proposal for " . ($payload['brand_name'] ?? 'Your Brand');
        
        $sections = [
            [
                'title' => 'Introduction',
                'content' => "Dear " . ($payload['creator_name'] ?? 'Creator') . ",\n\nWe are excited to propose a collaboration opportunity with your brand. Based on your campaign brief, we've crafted a proposal that aligns with your goals and our expertise.",
                'order' => 0,
            ],
            [
                'title' => 'Campaign Overview',
                'content' => "Campaign Brief:\n" . ($payload['campaign_brief'] ?? 'No brief provided') . "\n\nTone of Voice:\n" . ($payload['tone'] ?? 'Not specified'),
                'order' => 1,
            ],
            [
                'title' => 'Deliverables',
                'content' => "We propose the following deliverables:\n" . implode("\n", $payload['deliverables'] ?? ['Standard deliverables']),
                'order' => 2,
            ],
            [
                'title' => 'Timeline',
                'content' => "We estimate the campaign will take approximately 2-3 weeks to complete, including content creation, review, and posting.",
                'order' => 3,
            ],
            [
                'title' => 'Budget',
                'content' => "Based on the deliverables and scope of work, we propose a budget that reflects the high-quality content and engagement we'll provide.",
                'order' => 4,
            ],
        ];
        
        $priceEstimate = ($payload['budget_range'][1] ?? 1000) * 0.8;
        
        // Calculate token usage based on content length
        $textContent = "Proposal for {$payload['brand_name']}\n\n" . implode("\n\n", array_column($sections, 'content'));
        $metaContent = json_encode([
            'title' => $title,
            'sections' => $sections,
            'price_estimate' => $priceEstimate,
        ]);
        
        // Rough estimation: 1 token ≈ 4 characters
        $tokenUsage = (strlen($textContent) + strlen($metaContent)) / 4;
        
        return [
            'text' => $textContent,
            'meta' => [
                'title' => $title,
                'sections' => $sections,
                'price_estimate' => $priceEstimate,
            ],
            'token_usage' => $tokenUsage,
        ];
    }
}