<?php

namespace App\Jobs;

use App\Models\ScheduledPost;
use App\Models\ScheduledPostAttempt;
use App\Models\SocialAccount;
use App\Services\SocialProviders\ProviderFactory;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessScheduledPost implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $scheduledPost;

    /**
     * Create a new job instance.
     */
    public function __construct(ScheduledPost $scheduledPost)
    {
        $this->scheduledPost = $scheduledPost;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        try {
            // Log start of processing
            Log::info('Starting to process scheduled post', [
                'scheduled_post_id' => $this->scheduledPost->id,
                'brand_id' => $this->scheduledPost->brand_id,
                'status' => $this->scheduledPost->status
            ]);
            
            // Mark as publishing
            $this->scheduledPost->markPublishing();
            
            // Get the social account
            $socialAccount = SocialAccount::find($this->scheduledPost->social_account_id);
            
            if (!$socialAccount) {
                $this->scheduledPost->markFailed('Social account not found');
                Log::warning('Social account not found for scheduled post', [
                    'scheduled_post_id' => $this->scheduledPost->id,
                    'social_account_id' => $this->scheduledPost->social_account_id
                ]);
                return;
            }
            
            // Check if token is expired
            if ($socialAccount->isTokenExpired()) {
                $this->scheduledPost->markFailed('Social account token has expired');
                Log::warning('Social account token has expired', [
                    'scheduled_post_id' => $this->scheduledPost->id,
                    'social_account_id' => $socialAccount->id
                ]);
                return;
            }
            
            // Create post attempt record
            $attempt = ScheduledPostAttempt::create([
                'scheduled_post_id' => $this->scheduledPost->id,
                'social_account_id' => $socialAccount->id,
                'provider' => $socialAccount->provider,
            ]);
            
            // Get provider service
            $provider = ProviderFactory::make($socialAccount->provider);
            
            if (!$provider) {
                Log::warning('Provider not supported', [
                    'scheduled_post_id' => $this->scheduledPost->id,
                    'provider' => $socialAccount->provider
                ]);
                
                $attempt->markFailed('Provider not supported');
                $this->scheduledPost->markFailed('Provider not supported');
                return;
            }
            
            // Prepare request payload
            $payload = [
                'scheduled_post_id' => $this->scheduledPost->id,
                'caption' => $this->scheduledPost->caption,
                'media_refs' => $this->scheduledPost->getDecryptedMediaRefs(),
                'publish_time' => $this->scheduledPost->publish_time,
            ];
            
            $attempt->update([
                'request_payload' => $payload
            ]);
            
            // Post the content
            $result = $provider->postContent($this->scheduledPost, $socialAccount);
            
            Log::info('Content post result', [
                'scheduled_post_id' => $this->scheduledPost->id,
                'social_account_id' => $socialAccount->id,
                'provider' => $socialAccount->provider,
                'result_status' => $result['status'] ?? 'unknown'
            ]);
            
            // Update attempt with response
            $attempt->update([
                'response_summary' => $result
            ]);
            
            // Check if successful
            if (isset($result['status']) && $result['status'] === 'success') {
                $this->scheduledPost->markPosted($result['remote_post_id'] ?? null, $result);
                $attempt->markSuccessful($result);
                Log::info('Scheduled post marked as posted', [
                    'scheduled_post_id' => $this->scheduledPost->id
                ]);
            } else {
                $errorMessage = $result['message'] ?? 'Failed to post content';
                $attempt->markFailed($errorMessage, $result);
                $this->scheduledPost->markFailed($errorMessage);
                Log::warning('Scheduled post marked as failed', [
                    'scheduled_post_id' => $this->scheduledPost->id,
                    'error_message' => $errorMessage
                ]);
            }
            
            // Update attempt count
            $this->scheduledPost->increment('attempt_count');
            $this->scheduledPost->update(['last_attempt_at' => now()]);
        } catch (\Exception $e) {
            Log::error('Failed to process scheduled post: ' . $e->getMessage(), [
                'scheduled_post_id' => $this->scheduledPost->id,
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->scheduledPost->markFailed('Processing failed: ' . $e->getMessage());
        }
    }
}
