<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Campaign extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'agency_id',
        'brand_id',
        'name',
        'description',
        'objective',
        'deliverables',
        'audience',
        'start_date',
        'end_date',
        'budget',
        'currency',
        'status',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'budget' => 'decimal:2',
    ];
    
    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();
        
        // Encrypt budget when saving
        static::saving(function ($campaign) {
            if ($campaign->budget !== null) {
                $campaign->encrypted_budget = \Crypt::encryptString($campaign->budget);
            }
        });
        
        // Decrypt budget when retrieving
        static::retrieved(function ($campaign) {
            if ($campaign->encrypted_budget !== null) {
                try {
                    $campaign->budget = \Crypt::decryptString($campaign->encrypted_budget);
                } catch (\Exception $e) {
                    // If decryption fails, leave budget as is
                    \Log::error('Failed to decrypt campaign budget: ' . $e->getMessage(), [
                        'campaign_id' => $campaign->id,
                    ]);
                }
            }
        });
    }

    /**
     * Get the agency (user) that owns the campaign.
     */
    public function agency()
    {
        return $this->belongsTo(User::class, 'agency_id');
    }

    /**
     * Get the brand that owns the campaign.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }
    
    /**
     * Get the creators invited to the campaign.
     */
    public function creators()
    {
        return $this->belongsToMany(User::class, 'campaign_creators', 'campaign_id', 'creator_id')
            ->withPivot(['status', 'invitation_sent_at', 'responded_at', 'notes'])
            ->withTimestamps();
    }
    
    /**
     * Get the milestones for the campaign.
     */
    public function milestones()
    {
        return $this->hasMany(CampaignMilestone::class);
    }
    
    /**
     * Get the budget allocations for the campaign.
     */
    public function budgetAllocations()
    {
        return $this->hasMany(CampaignBudgetAllocation::class);
    }
}