<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CompetitorAlert extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'brand_id',
        'competitor_brand_id',
        'tenant_id',
        'alert_type',
        'metric',
        'severity',
        'title',
        'description',
        'data',
        'threshold_value',
        'current_value',
        'change_percentage',
        'competitor_post_id',
        'is_read',
        'is_dismissed',
        'read_at',
        'dismissed_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'data' => 'array',
        'threshold_value' => 'decimal:2',
        'current_value' => 'decimal:2',
        'change_percentage' => 'decimal:2',
        'is_read' => 'boolean',
        'is_dismissed' => 'boolean',
        'read_at' => 'datetime',
        'dismissed_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the alert.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the competitor brand for this alert.
     */
    public function competitorBrand()
    {
        return $this->belongsTo(CompetitorBrand::class);
    }

    /**
     * Get the tenant that owns the alert.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Get the related competitor post if applicable.
     */
    public function competitorPost()
    {
        return $this->belongsTo(CompetitorPost::class);
    }

    /**
     * Mark the alert as read.
     *
     * @return $this
     */
    public function markAsRead()
    {
        $this->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
        return $this;
    }

    /**
     * Mark the alert as unread.
     *
     * @return $this
     */
    public function markAsUnread()
    {
        $this->update([
            'is_read' => false,
            'read_at' => null,
        ]);
        return $this;
    }

    /**
     * Dismiss the alert.
     *
     * @return $this
     */
    public function dismiss()
    {
        $this->update([
            'is_dismissed' => true,
            'dismissed_at' => now(),
        ]);
        return $this;
    }

    /**
     * Restore a dismissed alert.
     *
     * @return $this
     */
    public function restore()
    {
        $this->update([
            'is_dismissed' => false,
            'dismissed_at' => null,
        ]);
        return $this;
    }

    /**
     * Check if the alert is critical.
     *
     * @return bool
     */
    public function isCritical()
    {
        return $this->severity === 'critical';
    }

    /**
     * Check if the alert is high severity.
     *
     * @return bool
     */
    public function isHighSeverity()
    {
        return $this->severity === 'high';
    }

    /**
     * Get the severity color for UI display.
     *
     * @return string
     */
    public function getSeverityColor()
    {
        return match($this->severity) {
            'critical' => 'red',
            'high' => 'orange',
            'medium' => 'yellow',
            'low' => 'blue',
            default => 'gray',
        };
    }

    /**
     * Scope a query to only include unread alerts.
     */
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    /**
     * Scope a query to only include read alerts.
     */
    public function scopeRead($query)
    {
        return $query->where('is_read', true);
    }

    /**
     * Scope a query to only include non-dismissed alerts.
     */
    public function scopeActive($query)
    {
        return $query->where('is_dismissed', false);
    }

    /**
     * Scope a query to filter by severity.
     */
    public function scopeSeverity($query, string $severity)
    {
        return $query->where('severity', $severity);
    }

    /**
     * Scope a query to filter by alert type.
     */
    public function scopeType($query, string $type)
    {
        return $query->where('alert_type', $type);
    }

    /**
     * Scope a query to filter by brand.
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope a query to filter by competitor.
     */
    public function scopeForCompetitor($query, $competitorBrandId)
    {
        return $query->where('competitor_brand_id', $competitorBrandId);
    }

    /**
     * Scope a query to get recent alerts.
     */
    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }
}