<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CompetitorBrand extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'brand_id',
        'tenant_id',
        'name',
        'slug',
        'description',
        'industry',
        'social_handles',
        'meta',
        'is_active',
        'last_synced_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'social_handles' => 'array',
        'meta' => 'array',
        'is_active' => 'boolean',
        'last_synced_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the competitor.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the tenant that owns the competitor.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Get the posts for the competitor.
     */
    public function posts()
    {
        return $this->hasMany(CompetitorPost::class);
    }

    /**
     * Get the insights for the competitor.
     */
    public function insights()
    {
        return $this->hasMany(CompetitorInsight::class);
    }

    /**
     * Get the alerts for the competitor.
     */
    public function alerts()
    {
        return $this->hasMany(CompetitorAlert::class);
    }

    /**
     * Get the social handle for a specific platform.
     *
     * @param string $platform
     * @return string|null
     */
    public function getSocialHandle(string $platform)
    {
        return $this->social_handles[$platform] ?? null;
    }

    /**
     * Set the social handle for a specific platform.
     *
     * @param string $platform
     * @param string $handle
     * @return $this
     */
    public function setSocialHandle(string $platform, string $handle)
    {
        $handles = $this->social_handles ?? [];
        $handles[$platform] = $handle;
        $this->social_handles = $handles;
        return $this;
    }

    /**
     * Scope a query to only include active competitors.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by brand.
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }
}