<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Crypt;

class CreatorSocialConnection extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'creator_id',
        'platform',
        'label',
        'client_id',
        'client_secret',
        'access_token',
        'refresh_token',
        'ad_account_id',
        'scopes',
        'redirect_uri',
        'platform_user_id',
        'platform_username',
        'platform_display_name',
        'profile_picture_url',
        'token_expires_at',
        'expiry_alert_sent',
        'status',
        'last_tested_at',
        'last_synced_at',
        'last_error',
        'use_for_ads',
        'ad_approval_at',
        'ad_approved_by',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'scopes' => 'array',
        'token_expires_at' => 'datetime',
        'expiry_alert_sent' => 'boolean',
        'last_tested_at' => 'datetime',
        'last_synced_at' => 'datetime',
        'use_for_ads' => 'boolean',
        'ad_approval_at' => 'datetime',
    ];

    protected $hidden = [
        'client_secret',
        'access_token',
        'refresh_token',
    ];

    /**
     * The attributes that should be encrypted.
     */
    protected $encrypted = [
        'client_id',
        'client_secret',
        'access_token',
        'refresh_token',
        'ad_account_id',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        // Automatically encrypt sensitive fields on save
        static::saving(function ($model) {
            foreach ($model->encrypted as $field) {
                if ($model->isDirty($field) && !empty($model->$field)) {
                    // Only encrypt if not already encrypted
                    if (!str_starts_with($model->$field, 'eyJpdiI6')) {
                        $model->attributes[$field] = Crypt::encryptString($model->$field);
                    }
                }
            }
        });

        // Track ad approval
        static::saving(function ($model) {
            if ($model->isDirty('use_for_ads') && $model->use_for_ads) {
                $model->ad_approval_at = now();
                $model->ad_approved_by = auth()->id();
            }
        });
    }

    /**
     * Get decrypted value for encrypted fields.
     */
    public function getAttribute($key)
    {
        $value = parent::getAttribute($key);

        if (in_array($key, $this->encrypted) && !empty($value)) {
            try {
                return Crypt::decryptString($value);
            } catch (\Exception $e) {
                return null;
            }
        }

        return $value;
    }

    /**
     * Get masked value for display (shows last 4 characters).
     */
    public function getMaskedAttribute($field)
    {
        $value = $this->getAttribute($field);
        
        if (empty($value)) {
            return null;
        }

        $length = strlen($value);
        if ($length <= 4) {
            return str_repeat('*', $length);
        }

        return str_repeat('*', $length - 4) . substr($value, -4);
    }

    /**
     * Check if token is expiring soon (within configured days).
     */
    public function isExpiringSoon($days = 7): bool
    {
        if (!$this->token_expires_at) {
            return false;
        }

        return $this->token_expires_at->lte(now()->addDays($days));
    }

    /**
     * Check if token is expired.
     */
    public function isExpired(): bool
    {
        if (!$this->token_expires_at) {
            return false;
        }

        return $this->token_expires_at->lte(now());
    }

    /**
     * Get the creator that owns the connection.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'creator_id');
    }

    /**
     * Get the user who created the connection.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated the connection.
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get the user who approved ad usage.
     */
    public function adApprovedBy()
    {
        return $this->belongsTo(User::class, 'ad_approved_by');
    }

    /**
     * Get shared access records for this connection.
     */
    public function sharedAccess()
    {
        return $this->hasMany(SharedSocialAccess::class, 'connection_id');
    }

    /**
     * Get active shared access records.
     */
    public function activeSharedAccess()
    {
        return $this->hasMany(SharedSocialAccess::class, 'connection_id')
            ->where('status', 'active')
            ->where('expires_at', '>', now());
    }

    /**
     * Scope to get only active connections.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'connected');
    }

    /**
     * Scope to get connections for ads.
     */
    public function scopeForAds($query)
    {
        return $query->where('use_for_ads', true);
    }

    /**
     * Scope to get expiring connections.
     */
    public function scopeExpiring($query, $days = 7)
    {
        return $query->where('token_expires_at', '<=', now()->addDays($days))
            ->where('token_expires_at', '>', now());
    }

    /**
     * Scope to get expired connections.
     */
    public function scopeExpired($query)
    {
        return $query->where('token_expires_at', '<=', now());
    }

    /**
     * Get platform display name.
     */
    public function getPlatformDisplayNameAttribute(): string
    {
        return match($this->platform) {
            'youtube' => 'YouTube',
            'instagram' => 'Instagram',
            'tiktok' => 'TikTok',
            'linkedin' => 'LinkedIn',
            'twitter' => 'X (Twitter)',
            'facebook' => 'Facebook',
            default => ucfirst($this->platform),
        };
    }

    /**
     * Get status badge color.
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'connected' => 'success',
            'expiring' => 'warning',
            'failed' => 'danger',
            'disconnected' => 'secondary',
            default => 'secondary',
        };
    }

    /**
     * Get platform icon class.
     */
    public function getPlatformIconAttribute(): string
    {
        return match($this->platform) {
            'youtube' => 'fab fa-youtube',
            'instagram' => 'fab fa-instagram',
            'tiktok' => 'fab fa-tiktok',
            'linkedin' => 'fab fa-linkedin',
            'twitter' => 'fab fa-x-twitter',
            'facebook' => 'fab fa-facebook',
            default => 'fas fa-link',
        };
    }

    /**
     * Check if connection can be shared with brands.
     */
    public function canBeShared(): bool
    {
        return $this->status === 'connected' && !$this->isExpired();
    }

    /**
     * Check if connection is shared with a specific brand.
     */
    public function isSharedWith(int $brandId): bool
    {
        return $this->activeSharedAccess()
            ->where('brand_id', $brandId)
            ->exists();
    }

    /**
     * Get display label or generate default.
     */
    public function getDisplayLabelAttribute(): string
    {
        if ($this->label) {
            return $this->label;
        }

        if ($this->platform_username) {
            return "{$this->platform_display_name} (@{$this->platform_username})";
        }

        return $this->platform_display_name;
    }
}