<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class KnowledgeBaseDocument extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'brand_id',
        'title',
        'document_type',
        'content',
        'content_summary',
        'metadata',
        'vector_embedding',
        'file_path',
        'file_type',
        'file_size',
        'status',
        'uploaded_by',
        'last_indexed_at',
    ];

    protected $casts = [
        'metadata' => 'array',
        'vector_embedding' => 'array',
        'last_indexed_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the document.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the user who uploaded the document.
     */
    public function uploader()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Scope to get only active documents.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Scope to filter by document type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('document_type', $type);
    }

    /**
     * Scope to search documents by content.
     */
    public function scopeSearch($query, string $searchTerm)
    {
        return $query->whereFullText(['title', 'content'], $searchTerm);
    }

    /**
     * Get file size in human-readable format.
     */
    public function getFileSizeHumanAttribute(): ?string
    {
        if (!$this->file_size) {
            return null;
        }

        $units = ['B', 'KB', 'MB', 'GB'];
        $size = $this->file_size;
        $unit = 0;

        while ($size >= 1024 && $unit < count($units) - 1) {
            $size /= 1024;
            $unit++;
        }

        return round($size, 2) . ' ' . $units[$unit];
    }

    /**
     * Check if document needs re-indexing.
     */
    public function needsReindexing(): bool
    {
        if (!$this->last_indexed_at) {
            return true;
        }

        return $this->updated_at->isAfter($this->last_indexed_at);
    }
}