<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PerformanceForecast extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'brand_id',
        'created_by',
        'campaign_ids',
        'social_account_ids',
        'start_date',
        'end_date',
        'forecast_days',
        'input_metrics',
        'scenario',
        'predicted_timeseries',
        'error_bands',
        'roi_estimates',
        'insights_text',
        'insights_meta',
        'status',
        'error_message',
        'completed_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'campaign_ids' => 'array',
        'social_account_ids' => 'array',
        'start_date' => 'date',
        'end_date' => 'date',
        'input_metrics' => 'array',
        'predicted_timeseries' => 'array',
        'error_bands' => 'array',
        'roi_estimates' => 'array',
        'insights_meta' => 'array',
        'completed_at' => 'datetime',
    ];

    /**
     * Get the brand that owns the forecast.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the user who created the forecast.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the campaigns associated with this forecast.
     */
    public function campaigns()
    {
        if (empty($this->campaign_ids)) {
            return collect();
        }
        
        return Campaign::whereIn('id', $this->campaign_ids)->get();
    }

    /**
     * Get the social accounts associated with this forecast.
     */
    public function socialAccounts()
    {
        if (empty($this->social_account_ids)) {
            return collect();
        }
        
        return SocialAccount::whereIn('id', $this->social_account_ids)->get();
    }

    /**
     * Check if the forecast is completed.
     *
     * @return bool
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Check if the forecast is pending.
     *
     * @return bool
     */
    public function isPending()
    {
        return $this->status === 'pending';
    }

    /**
     * Check if the forecast is processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return $this->status === 'processing';
    }

    /**
     * Check if the forecast has failed.
     *
     * @return bool
     */
    public function hasFailed()
    {
        return $this->status === 'failed';
    }

    /**
     * Mark the forecast as processing.
     *
     * @return $this
     */
    public function markAsProcessing()
    {
        $this->update(['status' => 'processing']);
        return $this;
    }

    /**
     * Mark the forecast as completed.
     *
     * @param array $results
     * @return $this
     */
    public function markAsCompleted(array $results)
    {
        $this->update([
            'status' => 'completed',
            'predicted_timeseries' => $results['predicted_timeseries'] ?? [],
            'error_bands' => $results['error_bands'] ?? [],
            'roi_estimates' => $results['roi_estimates'] ?? [],
            'insights_text' => $results['insights_text'] ?? null,
            'insights_meta' => $results['insights_meta'] ?? [],
            'completed_at' => now(),
        ]);
        
        return $this;
    }

    /**
     * Mark the forecast as failed.
     *
     * @param string $errorMessage
     * @return $this
     */
    public function markAsFailed(string $errorMessage)
    {
        $this->update([
            'status' => 'failed',
            'error_message' => $errorMessage,
        ]);
        
        return $this;
    }

    /**
     * Get actionable insights from the forecast.
     *
     * @return array
     */
    public function getActionableInsights()
    {
        if (empty($this->insights_meta)) {
            return [];
        }

        return $this->insights_meta['actionable_insights'] ?? [];
    }

    /**
     * Get predicted KPIs summary.
     *
     * @return array
     */
    public function getPredictedKPIs()
    {
        if (empty($this->roi_estimates)) {
            return [];
        }

        return [
            'predicted_roi' => $this->roi_estimates['roi'] ?? 0,
            'predicted_reach' => $this->roi_estimates['reach'] ?? 0,
            'predicted_conversions' => $this->roi_estimates['conversions'] ?? 0,
            'predicted_engagement_rate' => $this->roi_estimates['engagement_rate'] ?? 0,
        ];
    }

    /**
     * Scope a query to only include forecasts for a specific brand.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $brandId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeForBrand($query, $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope a query to only include completed forecasts.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope a query to only include pending forecasts.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }
}