<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;

class RazorpaySetting extends Model
{
    protected $fillable = [
        'key_id',
        'key_secret',
        'webhook_secret',
        'test_mode',
        'currency',
        'is_active',
        'metadata',
    ];

    protected $casts = [
        'test_mode' => 'boolean',
        'is_active' => 'boolean',
        'metadata' => 'array',
    ];

    protected $hidden = [
        'key_secret',
    ];

    /**
     * Get the decrypted key_id.
     */
    public function getKeyIdAttribute($value): ?string
    {
        if (!$value) {
            return null;
        }

        try {
            return Crypt::decryptString($value);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Set the encrypted key_id.
     */
    public function setKeyIdAttribute($value): void
    {
        if ($value) {
            $this->attributes['key_id'] = Crypt::encryptString($value);
        }
    }

    /**
     * Get the decrypted key_secret.
     */
    public function getKeySecretAttribute($value): ?string
    {
        if (!$value) {
            return null;
        }

        try {
            return Crypt::decryptString($value);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Set the encrypted key_secret.
     */
    public function setKeySecretAttribute($value): void
    {
        if ($value) {
            $this->attributes['key_secret'] = Crypt::encryptString($value);
        }
    }

    /**
     * Check if Razorpay is configured.
     */
    public function isConfigured(): bool
    {
        return !empty($this->key_id) && !empty($this->key_secret);
    }

    /**
     * Get the singleton instance.
     */
    public static function getInstance(): ?self
    {
        return self::first();
    }

    /**
     * Get or create the singleton instance.
     */
    public static function getOrCreate(): self
    {
        $setting = self::first();

        if (!$setting) {
            $setting = self::create([
                'test_mode' => true,
                'currency' => 'INR',
                'is_active' => false,
            ]);
        }

        return $setting;
    }

    /**
     * Get Razorpay API credentials.
     */
    public function getCredentials(): array
    {
        return [
            'key_id' => $this->key_id,
            'key_secret' => $this->key_secret,
        ];
    }

    /**
     * Check if test mode is enabled.
     */
    public function isTestMode(): bool
    {
        return $this->test_mode;
    }

    /**
     * Check if Razorpay is active.
     */
    public function isActive(): bool
    {
        return $this->is_active && $this->isConfigured();
    }
}