<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ReputationAuditLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'brand_id',
        'reputation_score_id',
        'reputation_driver_id',
        'user_id',
        'action',
        'description',
        'before_state',
        'after_state',
        'ip_address',
        'user_agent',
        'metadata',
    ];

    protected $casts = [
        'before_state' => 'array',
        'after_state' => 'array',
        'metadata' => 'array',
    ];

    /**
     * Get the brand that owns the audit log.
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the reputation score associated with this log.
     */
    public function reputationScore(): BelongsTo
    {
        return $this->belongsTo(ReputationScore::class);
    }

    /**
     * Get the reputation driver associated with this log.
     */
    public function reputationDriver(): BelongsTo
    {
        return $this->belongsTo(ReputationDriver::class);
    }

    /**
     * Get the user who performed the action.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the changes made.
     */
    public function getChangesAttribute(): array
    {
        if (!$this->before_state || !$this->after_state) {
            return [];
        }

        $changes = [];
        foreach ($this->after_state as $key => $value) {
            if (!isset($this->before_state[$key]) || $this->before_state[$key] !== $value) {
                $changes[$key] = [
                    'from' => $this->before_state[$key] ?? null,
                    'to' => $value,
                ];
            }
        }

        return $changes;
    }

    /**
     * Check if this is a manual action.
     */
    public function isManualAction(): bool
    {
        return in_array($this->action, ['manual_override', 'driver_resolved', 'settings_updated']);
    }

    /**
     * Check if this is a system action.
     */
    public function isSystemAction(): bool
    {
        return in_array($this->action, ['score_calculated', 'score_recalculated', 'alert_triggered']);
    }

    /**
     * Scope to get logs for a specific brand.
     */
    public function scopeForBrand($query, int $brandId)
    {
        return $query->where('brand_id', $brandId);
    }

    /**
     * Scope to get logs by action type.
     */
    public function scopeByAction($query, string $action)
    {
        return $query->where('action', $action);
    }

    /**
     * Scope to get manual actions.
     */
    public function scopeManualActions($query)
    {
        return $query->whereIn('action', ['manual_override', 'driver_resolved', 'settings_updated']);
    }

    /**
     * Scope to get system actions.
     */
    public function scopeSystemActions($query)
    {
        return $query->whereIn('action', ['score_calculated', 'score_recalculated', 'alert_triggered']);
    }

    /**
     * Scope to get recent logs.
     */
    public function scopeRecent($query, int $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }
}