<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ReputationScore extends Model
{
    use HasFactory;

    protected $fillable = [
        'brand_id',
        'score',
        'sentiment_score',
        'responsiveness_score',
        'trust_score',
        'compliance_score',
        'growth_score',
        'timeframe',
        'period_start',
        'period_end',
        'ai_job_id',
        'ai_explanation',
        'calculation_metadata',
        'previous_score',
        'score_change',
        'score_change_percentage',
        'status',
        'alert_triggered',
        'alert_reason',
    ];

    protected $casts = [
        'score' => 'decimal:2',
        'sentiment_score' => 'decimal:2',
        'responsiveness_score' => 'decimal:2',
        'trust_score' => 'decimal:2',
        'compliance_score' => 'decimal:2',
        'growth_score' => 'decimal:2',
        'previous_score' => 'decimal:2',
        'score_change' => 'decimal:2',
        'score_change_percentage' => 'decimal:2',
        'calculation_metadata' => 'array',
        'period_start' => 'datetime',
        'period_end' => 'datetime',
        'alert_triggered' => 'boolean',
    ];

    /**
     * Get the brand that owns the reputation score.
     */
    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Get the drivers for this reputation score.
     */
    public function drivers(): HasMany
    {
        return $this->hasMany(ReputationDriver::class);
    }

    /**
     * Get the recommendations for this reputation score.
     */
    public function recommendations(): HasMany
    {
        return $this->hasMany(ReputationRecommendation::class);
    }

    /**
     * Get the audit logs for this reputation score.
     */
    public function auditLogs(): HasMany
    {
        return $this->hasMany(ReputationAuditLog::class);
    }

    /**
     * Get positive drivers.
     */
    public function positiveDrivers(): HasMany
    {
        return $this->drivers()->where('type', 'positive')->orderByDesc('impact_score');
    }

    /**
     * Get negative drivers.
     */
    public function negativeDrivers(): HasMany
    {
        return $this->drivers()->where('type', 'negative')->orderByDesc('impact_score');
    }

    /**
     * Get unresolved negative drivers.
     */
    public function unresolvedNegativeDrivers(): HasMany
    {
        return $this->negativeDrivers()->where('is_resolved', false);
    }

    /**
     * Get the score grade (A-F).
     */
    public function getGradeAttribute(): string
    {
        $score = $this->score;
        
        if ($score >= 90) return 'A';
        if ($score >= 80) return 'B';
        if ($score >= 70) return 'C';
        if ($score >= 60) return 'D';
        return 'F';
    }

    /**
     * Get the score status (excellent, good, fair, poor).
     */
    public function getStatusLabelAttribute(): string
    {
        $score = $this->score;
        
        if ($score >= 85) return 'excellent';
        if ($score >= 70) return 'good';
        if ($score >= 50) return 'fair';
        return 'poor';
    }

    /**
     * Get the trend direction.
     */
    public function getTrendAttribute(): string
    {
        if ($this->score_change > 0) return 'up';
        if ($this->score_change < 0) return 'down';
        return 'stable';
    }

    /**
     * Check if score has significantly dropped.
     */
    public function hasSignificantDrop(): bool
    {
        return $this->score_change_percentage <= -10;
    }

    /**
     * Get AI explanation as array.
     */
    public function getAiExplanationArrayAttribute(): array
    {
        if (!$this->ai_explanation) {
            return [];
        }

        // Split by bullet points or newlines
        $lines = preg_split('/[\r\n]+/', $this->ai_explanation);
        return array_filter(array_map('trim', $lines));
    }

    /**
     * Scope to get latest score for a brand.
     */
    public function scopeLatestForBrand($query, int $brandId)
    {
        return $query->where('brand_id', $brandId)
            ->where('status', 'completed')
            ->latest();
    }

    /**
     * Scope to get scores with alerts.
     */
    public function scopeWithAlerts($query)
    {
        return $query->where('alert_triggered', true);
    }

    /**
     * Scope to get scores by timeframe.
     */
    public function scopeByTimeframe($query, string $timeframe)
    {
        return $query->where('timeframe', $timeframe);
    }
}