<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Crypt;

class SocialAccount extends Model
{
    use HasFactory;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'brand_id',
        'tenant_id',
        'provider',
        'provider_user_id',
        'access_token_encrypted',
        'refresh_token_encrypted',
        'token_expires_at',
        'meta',
        'connected_at',
        'last_synced_at',
        'is_demo',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'meta' => 'array',
        'token_expires_at' => 'datetime',
        'connected_at' => 'datetime',
        'last_synced_at' => 'datetime',
        'is_demo' => 'boolean',
    ];

    /**
     * Get the user that owns the social account.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    /**
     * Get the brand that owns the social account.
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }
    
    /**
     * Get the tenant that owns the social account.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Get the posts for the social account.
     */
    public function posts()
    {
        return $this->hasMany(SocialPost::class);
    }

    /**
     * Get the sync errors for the social account.
     */
    public function syncErrors()
    {
        return $this->hasMany(SocialSyncError::class);
    }

    /**
     * Check if the token is expired.
     *
     * @return bool
     */
    public function isTokenExpired()
    {
        // If it's a demo account, never consider it expired
        if ($this->is_demo) {
            return false;
        }

        // If there's no expiry time, assume it's expired
        if (!$this->token_expires_at) {
            return true;
        }

        // Check if the token has expired
        return $this->token_expires_at->isPast();
    }

    /**
     * Get the decrypted access token.
     *
     * @return string|null
     */
    public function getAccessToken()
    {
        if ($this->is_demo) {
            return null;
        }

        try {
            return Crypt::decryptString($this->access_token_encrypted);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get the decrypted refresh token.
     *
     * @return string|null
     */
    public function getRefreshToken()
    {
        if ($this->is_demo || !$this->refresh_token_encrypted) {
            return null;
        }

        try {
            return Crypt::decryptString($this->refresh_token_encrypted);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Set the encrypted access token.
     *
     * @param string $token
     * @return $this
     */
    public function setAccessToken($token)
    {
        $this->access_token_encrypted = Crypt::encryptString($token);
        return $this;
    }

    /**
     * Set the encrypted refresh token.
     *
     * @param string|null $token
     * @return $this
     */
    public function setRefreshToken($token)
    {
        $this->refresh_token_encrypted = $token ? Crypt::encryptString($token) : null;
        return $this;
    }
    
    /**
     * Check if the account is a demo account.
     *
     * @return bool
     */
    public function isDemo()
    {
        return $this->is_demo === true;
    }
    
    /**
     * Get the decrypted access token with try/catch and fallback.
     *
     * @return string|null
     */
    public function getDecryptedToken()
    {
        // For demo accounts, return null
        if ($this->isDemo()) {
            return null;
        }
        
        // If no encrypted token, return null
        if (!$this->access_token_encrypted) {
            return null;
        }
        
        try {
            return Crypt::decryptString($this->access_token_encrypted);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::warning('Failed to decrypt access token for account ' . $this->id . ': ' . $e->getMessage());
            return null;
        }
    }
}
