<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
       'name',
       'email',
       'password',
       'tenant_id',
       'brand_id',
       'influencer_type',
       'influencer_type_custom',
       'meta',
   ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
           'email_verified_at' => 'datetime',
           'password' => 'hashed',
           'meta' => 'array',
       ];
    }
    
    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }
    
    public function brands()
    {
        return $this->belongsToMany(Brand::class, 'agency_brand');
    }
    
    /**
     * Get the user's bio from meta field.
     *
     * @return string|null
     */
    public function getBioAttribute()
    {
        return $this->meta['bio'] ?? null;
    }
    
    /**
     * Get the user's avatar from meta field.
     *
     * @return string|null
     */
    public function getAvatarAttribute()
    {
        return $this->meta['avatar'] ?? null;
    }
    
    /**
     * Get the full URL to the user's avatar.
     *
     * @return string|null
     */
    public function getAvatarUrlAttribute()
    {
        if (empty($this->avatar)) {
            return null;
        }
        
        return asset('storage/' . $this->avatar);
    }
    
    /**
     * Get the social accounts for the user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function socialAccounts()
    {
        return $this->hasMany(SocialAccount::class, 'user_id');
    }
    
    /**
    * Get the social aggregates for the user.
    *
    * @return \Illuminate\Database\Eloquent\Relations\HasMany
    */
   public function socialAggregates()
   {
       return $this->hasMany(SocialAggregate::class, 'user_id');
   }
   
   /**
    * Get the contracts created by the user.
    */
   public function contracts()
   {
       return $this->hasMany(Contract::class, 'creator_id');
   }
   
   /**
    * Get the payments made to the user.
    */
   public function payments()
   {
       return $this->hasMany(Payment::class, 'creator_id');
   }
   
   /**
    * Get the proposal transitions made by the user.
    */
   public function proposalTransitions()
   {
       return $this->hasMany(ProposalTransition::class, 'changed_by');
   }
   
   /**
    * Get the user's active subscription.
    */
   public function subscription()
   {
       return $this->hasOne(Subscription::class)->latestOfMany();
   }
   
   /**
    * Get all subscriptions for the user.
    */
   public function subscriptions()
   {
       return $this->hasMany(Subscription::class);
   }
   
   /**
    * Check if user has an active subscription.
    */
   public function hasActiveSubscription(): bool
   {
       return $this->subscription && $this->subscription->isValid();
   }
   
   /**
    * Check if user can use a specific feature.
    */
   public function canUseFeature(string $feature): bool
   {
       if (!$this->subscription) {
           return false;
       }
       
       return $this->subscription->canUseFeature($feature);
   }
   
   /**
    * Get the user's current plan.
    */
   public function currentPlan()
   {
       return $this->subscription ? $this->subscription->planModel : null;
   }
}
