<?php

namespace App\Policies;

use App\Models\BudgetScenario;
use App\Models\User;

class BudgetScenarioPolicy
{
    /**
     * Determine if the user can view any budget scenarios
     */
    public function viewAny(User $user): bool
    {
        return $user->role === 'brand' || $user->role === 'admin';
    }

    /**
     * Determine if the user can view the budget scenario
     */
    public function view(User $user, BudgetScenario $scenario): bool
    {
        // Admin can view all scenarios
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only view their own scenarios
        return $user->role === 'brand' && $user->id === $scenario->brand_id;
    }

    /**
     * Determine if the user can create budget scenarios
     */
    public function create(User $user): bool
    {
        return $user->role === 'brand' || $user->role === 'admin';
    }

    /**
     * Determine if the user can update the budget scenario
     */
    public function update(User $user, BudgetScenario $scenario): bool
    {
        // Admin can update all scenarios
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only update their own scenarios
        return $user->role === 'brand' && $user->id === $scenario->brand_id;
    }

    /**
     * Determine if the user can delete the budget scenario
     */
    public function delete(User $user, BudgetScenario $scenario): bool
    {
        // Admin can delete all scenarios
        if ($user->role === 'admin') {
            return true;
        }

        // Brand can only delete their own scenarios
        return $user->role === 'brand' && $user->id === $scenario->brand_id;
    }

    /**
     * Determine if the user can apply the budget scenario
     */
    public function apply(User $user, BudgetScenario $scenario): bool
    {
        return $this->update($user, $scenario);
    }
}