<?php

namespace App\Services;

use App\Models\CompetitorInsight;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class CompetitorAIInsightsService
{
    protected string $apiKey;
    protected string $apiUrl;

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key', env('OPENAI_API_KEY'));
        $this->apiUrl = 'https://api.openai.com/v1/chat/completions';
    }

    /**
     * Generate AI-powered recommendations based on competitor insights.
     *
     * @param CompetitorInsight $insight
     * @return array
     */
    public function generateRecommendations(CompetitorInsight $insight): array
    {
        try {
            $prompt = $this->buildRecommendationPrompt($insight);
            
            $response = Http::withHeaders([
                'Authorization' => "Bearer {$this->apiKey}",
                'Content-Type' => 'application/json',
            ])->timeout(60)->post($this->apiUrl, [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a social media marketing expert analyzing competitor performance data. Provide actionable, specific recommendations based on the data provided.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 1500,
            ]);

            if ($response->successful()) {
                $content = $response->json()['choices'][0]['message']['content'] ?? '';
                return $this->parseRecommendations($content);
            }

            Log::error('AI API request failed', ['response' => $response->body()]);
            return $this->getFallbackRecommendations($insight);

        } catch (\Exception $e) {
            Log::error("Failed to generate AI recommendations: {$e->getMessage()}");
            return $this->getFallbackRecommendations($insight);
        }
    }

    /**
     * Generate executive summary using AI.
     *
     * @param CompetitorInsight $insight
     * @return string
     */
    public function generateSummary(CompetitorInsight $insight): string
    {
        try {
            $prompt = $this->buildSummaryPrompt($insight);
            
            $response = Http::withHeaders([
                'Authorization' => "Bearer {$this->apiKey}",
                'Content-Type' => 'application/json',
            ])->timeout(60)->post($this->apiUrl, [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a social media analyst creating executive summaries. Be concise, data-driven, and highlight key insights.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'temperature' => 0.7,
                'max_tokens' => 500,
            ]);

            if ($response->successful()) {
                return $response->json()['choices'][0]['message']['content'] ?? '';
            }

            return $this->getFallbackSummary($insight);

        } catch (\Exception $e) {
            Log::error("Failed to generate AI summary: {$e->getMessage()}");
            return $this->getFallbackSummary($insight);
        }
    }

    /**
     * Build prompt for generating recommendations.
     *
     * @param CompetitorInsight $insight
     * @return string
     */
    protected function buildRecommendationPrompt(CompetitorInsight $insight): string
    {
        $competitor = $insight->competitorBrand;
        $comparison = $insight->comparison_data ?? [];
        $contentStrategy = $insight->content_strategy ?? [];
        $engagementPatterns = $insight->engagement_patterns ?? [];
        $topPosts = $insight->top_posts ?? [];

        $prompt = "Analyze the following competitor performance data and provide 2-3 specific, actionable recommendations:\n\n";
        
        $prompt .= "Competitor: {$competitor->name}\n";
        $prompt .= "Analysis Period: {$insight->start_date->format('M d, Y')} to {$insight->end_date->format('M d, Y')}\n\n";

        // Add comparison data
        if (!empty($comparison)) {
            $prompt .= "Performance Comparison:\n";
            foreach ($comparison as $metric => $data) {
                if (isset($data['competitor'], $data['brand'], $data['difference_percentage'])) {
                    $prompt .= "- {$metric}: Competitor {$data['competitor']}, Your Brand {$data['brand']} ({$data['difference_percentage']}% difference)\n";
                }
            }
            $prompt .= "\n";
        }

        // Add content strategy insights
        if (!empty($contentStrategy)) {
            $prompt .= "Competitor Content Strategy:\n";
            if (isset($contentStrategy['post_types'])) {
                $prompt .= "- Post Types: " . json_encode($contentStrategy['post_types']) . "\n";
            }
            if (isset($contentStrategy['best_performing_type'])) {
                $prompt .= "- Best Performing Type: {$contentStrategy['best_performing_type']}\n";
            }
            $prompt .= "\n";
        }

        // Add engagement patterns
        if (!empty($engagementPatterns)) {
            $prompt .= "Engagement Patterns:\n";
            if (isset($engagementPatterns['best_posting_hour'])) {
                $prompt .= "- Best Posting Hour: {$engagementPatterns['best_posting_hour']}:00\n";
            }
            if (isset($engagementPatterns['best_posting_day'])) {
                $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                $dayName = $days[$engagementPatterns['best_posting_day']] ?? 'Unknown';
                $prompt .= "- Best Posting Day: {$dayName}\n";
            }
            $prompt .= "\n";
        }

        $prompt .= "Based on this data, provide specific recommendations on:\n";
        $prompt .= "1. Content strategy improvements\n";
        $prompt .= "2. Posting schedule optimization\n";
        $prompt .= "3. Engagement tactics to implement\n\n";
        $prompt .= "Format each recommendation as a JSON object with 'title', 'description', and 'priority' (high/medium/low) fields.";

        return $prompt;
    }

    /**
     * Build prompt for generating summary.
     *
     * @param CompetitorInsight $insight
     * @return string
     */
    protected function buildSummaryPrompt(CompetitorInsight $insight): string
    {
        $competitor = $insight->competitorBrand;
        $metrics = $insight->competitor_metrics ?? [];
        $comparison = $insight->comparison_data ?? [];

        $prompt = "Create a concise executive summary (2-3 paragraphs) of the following competitor analysis:\n\n";
        
        $prompt .= "Competitor: {$competitor->name}\n";
        $prompt .= "Period: {$insight->start_date->format('M d, Y')} to {$insight->end_date->format('M d, Y')}\n\n";

        if (!empty($metrics)) {
            $prompt .= "Key Metrics:\n";
            foreach ($metrics as $metric => $value) {
                $prompt .= "- {$metric}: {$value}\n";
            }
            $prompt .= "\n";
        }

        if (!empty($comparison)) {
            $prompt .= "Performance vs Your Brand:\n";
            $engagementComp = $comparison['avg_engagement_rate'] ?? null;
            if ($engagementComp) {
                $prompt .= "- Engagement Rate: {$engagementComp['difference_percentage']}% " . 
                          ($engagementComp['competitor_ahead'] ? 'ahead' : 'behind') . "\n";
            }
        }

        $prompt .= "\nProvide a professional summary highlighting the most important findings and their implications.";

        return $prompt;
    }

    /**
     * Parse AI-generated recommendations into structured format.
     *
     * @param string $content
     * @return array
     */
    protected function parseRecommendations(string $content): array
    {
        // Try to extract JSON objects from the response
        preg_match_all('/\{[^}]+\}/', $content, $matches);
        
        $recommendations = [];
        foreach ($matches[0] as $match) {
            try {
                $rec = json_decode($match, true);
                if ($rec && isset($rec['title'], $rec['description'])) {
                    $recommendations[] = [
                        'title' => $rec['title'],
                        'description' => $rec['description'],
                        'priority' => $rec['priority'] ?? 'medium',
                    ];
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        // If JSON parsing failed, try to extract recommendations from text
        if (empty($recommendations)) {
            $lines = explode("\n", $content);
            $currentRec = null;
            
            foreach ($lines as $line) {
                $line = trim($line);
                if (preg_match('/^\d+\./', $line) || preg_match('/^-/', $line)) {
                    if ($currentRec) {
                        $recommendations[] = $currentRec;
                    }
                    $currentRec = [
                        'title' => $line,
                        'description' => '',
                        'priority' => 'medium',
                    ];
                } elseif ($currentRec && !empty($line)) {
                    $currentRec['description'] .= $line . ' ';
                }
            }
            
            if ($currentRec) {
                $recommendations[] = $currentRec;
            }
        }

        return array_slice($recommendations, 0, 3); // Return max 3 recommendations
    }

    /**
     * Get fallback recommendations when AI fails.
     *
     * @param CompetitorInsight $insight
     * @return array
     */
    protected function getFallbackRecommendations(CompetitorInsight $insight): array
    {
        $recommendations = [];
        $comparison = $insight->comparison_data ?? [];

        // Engagement rate recommendation
        if (isset($comparison['avg_engagement_rate'])) {
            $engagementDiff = $comparison['avg_engagement_rate']['difference_percentage'] ?? 0;
            if ($engagementDiff > 20) {
                $recommendations[] = [
                    'title' => 'Improve Engagement Rate',
                    'description' => "Competitor is achieving {$engagementDiff}% higher engagement rate. Consider analyzing their content style, posting times, and interaction strategies.",
                    'priority' => 'high',
                ];
            }
        }

        // Posting frequency recommendation
        if (isset($comparison['posting_frequency'])) {
            $freqDiff = $comparison['posting_frequency']['difference_percentage'] ?? 0;
            if (abs($freqDiff) > 30) {
                $action = $freqDiff > 0 ? 'increase' : 'optimize';
                $recommendations[] = [
                    'title' => 'Adjust Posting Frequency',
                    'description' => "Competitor posts {$freqDiff}% more frequently. Consider {$action}ing your posting schedule to maintain visibility.",
                    'priority' => 'medium',
                ];
            }
        }

        // Content strategy recommendation
        $contentStrategy = $insight->content_strategy ?? [];
        if (isset($contentStrategy['best_performing_type'])) {
            $recommendations[] = [
                'title' => 'Leverage High-Performing Content Types',
                'description' => "Competitor sees best results with {$contentStrategy['best_performing_type']} content. Consider incorporating more of this content type into your strategy.",
                'priority' => 'medium',
            ];
        }

        return array_slice($recommendations, 0, 3);
    }

    /**
     * Get fallback summary when AI fails.
     *
     * @param CompetitorInsight $insight
     * @return string
     */
    protected function getFallbackSummary(CompetitorInsight $insight): string
    {
        $competitor = $insight->competitorBrand;
        $metrics = $insight->competitor_metrics ?? [];
        
        $summary = "Analysis of {$competitor->name} from {$insight->start_date->format('M d, Y')} to {$insight->end_date->format('M d, Y')}. ";
        
        if (isset($metrics['total_posts'])) {
            $summary .= "The competitor published {$metrics['total_posts']} posts during this period. ";
        }
        
        if (isset($metrics['avg_engagement_rate'])) {
            $summary .= "Their average engagement rate was {$metrics['avg_engagement_rate']}%. ";
        }
        
        $comparison = $insight->comparison_data ?? [];
        if (isset($comparison['avg_engagement_rate']['competitor_ahead'])) {
            $ahead = $comparison['avg_engagement_rate']['competitor_ahead'];
            $diff = abs($comparison['avg_engagement_rate']['difference_percentage'] ?? 0);
            $summary .= $ahead 
                ? "They are performing {$diff}% better than your brand in engagement. "
                : "Your brand is performing {$diff}% better in engagement. ";
        }

        return $summary;
    }
}