<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class PythonApiService
{
    protected $baseUrl;
    protected $apiKey;
    protected $timeout;
    protected $retryTimes;
    protected $retryDelay;

    public function __construct()
    {
        $this->baseUrl = config('python.api.base_url');
        $this->apiKey = config('python.api.api_key');
        $this->timeout = config('python.api.timeout', 30);
        $this->retryTimes = config('python.api.retry_times', 3);
        $this->retryDelay = config('python.api.retry_delay', 1000);
    }

    /**
     * Create an AI job in Python microservice
     *
     * @param string $jobType
     * @param string $tenantId
     * @param string $ownerId
     * @param array $payload
     * @param string|null $callbackUrl
     * @param string|null $idempotencyKey
     * @return array
     */
    public function createAiJob(
        string $jobType,
        string $tenantId,
        string $ownerId,
        array $payload,
        ?string $callbackUrl = null,
        ?string $idempotencyKey = null
    ): array {
        $endpoint = config('python.endpoints.ai_jobs');
        
        $data = [
            'job_type' => $jobType,
            'tenant_id' => $tenantId,
            'owner_id' => $ownerId,
            'payload' => $payload,
        ];

        if ($callbackUrl) {
            $data['callback_url'] = $callbackUrl;
        }

        if ($idempotencyKey) {
            $data['idempotency_key'] = $idempotencyKey;
        }

        return $this->post($endpoint, $data);
    }

    /**
     * Get AI job status
     *
     * @param string $jobId
     * @return array
     */
    public function getAiJobStatus(string $jobId): array
    {
        $endpoint = str_replace('{job_id}', $jobId, config('python.endpoints.ai_job_status'));
        return $this->get($endpoint);
    }

    /**
     * Publish content to social media
     *
     * @param string $platform
     * @param string $tenantId
     * @param string $ownerId
     * @param array $content
     * @param string|null $callbackUrl
     * @return array
     */
    public function publishToSocial(
        string $platform,
        string $tenantId,
        string $ownerId,
        array $content,
        ?string $callbackUrl = null
    ): array {
        $endpoint = config('python.endpoints.publish');
        
        $data = [
            'platform' => $platform,
            'tenant_id' => $tenantId,
            'owner_id' => $ownerId,
            'content' => $content,
        ];

        if ($callbackUrl) {
            $data['callback_url'] = $callbackUrl;
        }

        return $this->post($endpoint, $data);
    }

    /**
     * Get publish job status
     *
     * @param string $jobId
     * @return array
     */
    public function getPublishStatus(string $jobId): array
    {
        $endpoint = str_replace('{job_id}', $jobId, config('python.endpoints.publish_status'));
        return $this->get($endpoint);
    }

    /**
     * Ingest documents for RAG
     *
     * @param string $tenantId
     * @param array $documents
     * @param string $documentType
     * @return array
     */
    public function ingestDocuments(string $tenantId, array $documents, string $documentType = 'general'): array
    {
        $endpoint = config('python.endpoints.rag_ingest');
        
        $data = [
            'tenant_id' => $tenantId,
            'documents' => $documents,
            'document_type' => $documentType,
        ];

        return $this->post($endpoint, $data);
    }

    /**
     * Query RAG system
     *
     * @param string $tenantId
     * @param string $query
     * @param int $topK
     * @param bool $includeSources
     * @return array
     */
    public function queryRag(
        string $tenantId,
        string $query,
        int $topK = 5,
        bool $includeSources = true
    ): array {
        $endpoint = config('python.endpoints.rag_query');
        
        $data = [
            'tenant_id' => $tenantId,
            'query' => $query,
            'top_k' => $topK,
            'include_sources' => $includeSources,
        ];

        return $this->post($endpoint, $data);
    }

    /**
     * Process reel/video
     *
     * @param string $tenantId
     * @param string $videoUrl
     * @param array $options
     * @return array
     */
    public function processReel(string $tenantId, string $videoUrl, array $options = []): array
    {
        $endpoint = config('python.endpoints.reels_process');
        
        $data = array_merge([
            'tenant_id' => $tenantId,
            'video_url' => $videoUrl,
        ], $options);

        return $this->post($endpoint, $data);
    }

    /**
     * Check Python service health
     *
     * @return bool
     */
    public function checkHealth(): bool
    {
        try {
            $endpoint = config('python.endpoints.health');
            $response = $this->get($endpoint);
            return isset($response['status']) && $response['status'] === 'healthy';
        } catch (\Exception $e) {
            Log::error('Python service health check failed', [
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Verify callback signature
     *
     * @param string $payload
     * @param string $signature
     * @return bool
     */
    public function verifyCallbackSignature(string $payload, string $signature): bool
    {
        if (!config('python.callback.verify_signature')) {
            return true;
        }

        $secret = config('python.callback.secret');
        $expectedSignature = hash_hmac('sha256', $payload, $secret);
        
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Make GET request to Python API
     *
     * @param string $endpoint
     * @param array $query
     * @return array
     */
    protected function get(string $endpoint, array $query = []): array
    {
        $url = $this->baseUrl . $endpoint;

        if (config('python.logging.enabled')) {
            Log::channel(config('python.logging.channel'))->info('Python API GET request', [
                'url' => $url,
                'query' => $query,
            ]);
        }

        $response = Http::withHeaders([
            'X-API-Key' => $this->apiKey,
            'Accept' => 'application/json',
        ])
        ->timeout($this->timeout)
        ->retry($this->retryTimes, $this->retryDelay)
        ->get($url, $query);

        return $this->handleResponse($response);
    }

    /**
     * Make POST request to Python API
     *
     * @param string $endpoint
     * @param array $data
     * @return array
     */
    protected function post(string $endpoint, array $data): array
    {
        $url = $this->baseUrl . $endpoint;

        if (config('python.logging.enabled')) {
            Log::channel(config('python.logging.channel'))->info('Python API POST request', [
                'url' => $url,
                'data' => $data,
            ]);
        }

        $response = Http::withHeaders([
            'X-API-Key' => $this->apiKey,
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ])
        ->timeout($this->timeout)
        ->retry($this->retryTimes, $this->retryDelay)
        ->post($url, $data);

        return $this->handleResponse($response);
    }

    /**
     * Handle API response
     *
     * @param \Illuminate\Http\Client\Response $response
     * @return array
     * @throws \Exception
     */
    protected function handleResponse($response): array
    {
        if ($response->successful()) {
            $data = $response->json();
            
            if (config('python.logging.enabled')) {
                Log::channel(config('python.logging.channel'))->info('Python API response', [
                    'status' => $response->status(),
                    'data' => $data,
                ]);
            }
            
            return $data;
        }

        $error = $response->json();
        
        Log::error('Python API request failed', [
            'status' => $response->status(),
            'error' => $error,
        ]);

        throw new \Exception(
            $error['detail'] ?? $error['error'] ?? 'Python API request failed',
            $response->status()
        );
    }
}