<?php

namespace App\Services;

use App\Models\KnowledgeBaseDocument;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Exception;

class RAGService
{
    protected string $apiKey;
    protected string $apiUrl;
    protected string $model;
    protected string $embeddingModel;

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
        $this->apiUrl = config('services.openai.api_url', 'https://api.openai.com/v1/chat/completions');
        $this->model = config('services.openai.model', 'gpt-4');
        $this->embeddingModel = config('services.openai.embedding_model', 'text-embedding-ada-002');
    }

    /**
     * Generate answer using RAG approach.
     *
     * @param string $question Customer question
     * @param int $brandId Brand ID for knowledge base
     * @param array $conversationHistory Previous messages for context
     * @return array ['answer' => string, 'citations' => array, 'confidence_score' => float, 'requires_escalation' => bool]
     */
    public function generateAnswer(
        string $question,
        int $brandId,
        array $conversationHistory = []
    ): array {
        try {
            // Step 1: Retrieve relevant documents
            $relevantDocs = $this->retrieveRelevantDocuments($question, $brandId);

            if (empty($relevantDocs)) {
                return $this->handleNoRelevantDocuments($question);
            }

            // Step 2: Build context from retrieved documents
            $context = $this->buildContext($relevantDocs);

            // Step 3: Generate answer using LLM
            $response = $this->generateLLMResponse($question, $context, $conversationHistory);

            // Step 4: Calculate confidence score
            $confidenceScore = $this->calculateConfidenceScore($response, $relevantDocs);

            // Step 5: Prepare citations
            $citations = $this->prepareCitations($relevantDocs);

            return [
                'answer' => $response['answer'],
                'citations' => $citations,
                'confidence_score' => $confidenceScore,
                'requires_escalation' => $confidenceScore < 70.0,
                'metadata' => [
                    'documents_retrieved' => count($relevantDocs),
                    'model_used' => $this->model,
                ],
            ];

        } catch (Exception $e) {
            Log::error('RAG answer generation failed', [
                'error' => $e->getMessage(),
                'question' => $question,
                'brand_id' => $brandId,
            ]);

            return $this->handleError($e);
        }
    }

    /**
     * Retrieve relevant documents from knowledge base.
     */
    protected function retrieveRelevantDocuments(string $question, int $brandId, int $limit = 5): array
    {
        // Generate embedding for the question
        $questionEmbedding = $this->generateEmbedding($question);

        // Get active documents for the brand
        $documents = KnowledgeBaseDocument::where('brand_id', $brandId)
            ->active()
            ->get();

        if ($documents->isEmpty()) {
            return [];
        }

        // Calculate similarity scores
        $scoredDocs = [];
        foreach ($documents as $doc) {
            if (!$doc->vector_embedding) {
                continue;
            }

            $similarity = $this->cosineSimilarity(
                $questionEmbedding,
                $doc->vector_embedding
            );

            $scoredDocs[] = [
                'document' => $doc,
                'similarity' => $similarity,
            ];
        }

        // Sort by similarity and get top results
        usort($scoredDocs, fn($a, $b) => $b['similarity'] <=> $a['similarity']);

        return array_slice($scoredDocs, 0, $limit);
    }

    /**
     * Generate embedding for text.
     */
    public function generateEmbedding(string $text): array
    {
        $cacheKey = 'embedding_' . md5($text);

        return Cache::remember($cacheKey, 3600, function () use ($text) {
            try {
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Content-Type' => 'application/json',
                ])->timeout(30)->post('https://api.openai.com/v1/embeddings', [
                    'model' => $this->embeddingModel,
                    'input' => $text,
                ]);

                if (!$response->successful()) {
                    throw new Exception('Embedding API request failed: ' . $response->body());
                }

                $result = $response->json();
                return $result['data'][0]['embedding'] ?? [];

            } catch (Exception $e) {
                Log::error('Embedding generation failed', [
                    'error' => $e->getMessage(),
                    'text_length' => strlen($text),
                ]);
                throw $e;
            }
        });
    }

    /**
     * Calculate cosine similarity between two vectors.
     */
    protected function cosineSimilarity(array $vec1, array $vec2): float
    {
        if (count($vec1) !== count($vec2)) {
            return 0.0;
        }

        $dotProduct = 0.0;
        $magnitude1 = 0.0;
        $magnitude2 = 0.0;

        for ($i = 0; $i < count($vec1); $i++) {
            $dotProduct += $vec1[$i] * $vec2[$i];
            $magnitude1 += $vec1[$i] * $vec1[$i];
            $magnitude2 += $vec2[$i] * $vec2[$i];
        }

        $magnitude1 = sqrt($magnitude1);
        $magnitude2 = sqrt($magnitude2);

        if ($magnitude1 == 0 || $magnitude2 == 0) {
            return 0.0;
        }

        return $dotProduct / ($magnitude1 * $magnitude2);
    }

    /**
     * Build context from retrieved documents.
     */
    protected function buildContext(array $scoredDocs): string
    {
        $context = "Relevant information from knowledge base:\n\n";

        foreach ($scoredDocs as $index => $item) {
            $doc = $item['document'];
            $context .= "Document " . ($index + 1) . " (Relevance: " . round($item['similarity'] * 100, 1) . "%):\n";
            $context .= "Title: " . $doc->title . "\n";
            $context .= "Content: " . $doc->content . "\n\n";
        }

        return $context;
    }

    /**
     * Generate LLM response.
     */
    protected function generateLLMResponse(
        string $question,
        string $context,
        array $conversationHistory
    ): array {
        $systemPrompt = "You are a helpful customer support assistant. Use the provided knowledge base information to answer customer questions accurately and professionally. If the information is not in the knowledge base, politely say so and suggest contacting a human agent. Always cite your sources when providing information.";

        $messages = [
            ['role' => 'system', 'content' => $systemPrompt],
        ];

        // Add conversation history
        foreach ($conversationHistory as $msg) {
            $messages[] = [
                'role' => $msg['role'] ?? 'user',
                'content' => $msg['content'],
            ];
        }

        // Add current question with context
        $userMessage = "Context:\n{$context}\n\nCustomer Question: {$question}\n\nPlease provide a helpful answer based on the context above.";
        $messages[] = ['role' => 'user', 'content' => $userMessage];

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Content-Type' => 'application/json',
        ])->timeout(60)->post($this->apiUrl, [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => 0.3,
            'max_tokens' => 1000,
        ]);

        if (!$response->successful()) {
            throw new Exception('LLM API request failed: ' . $response->body());
        }

        $result = $response->json();
        $answer = $result['choices'][0]['message']['content'] ?? '';

        return [
            'answer' => $answer,
            'usage' => $result['usage'] ?? [],
        ];
    }

    /**
     * Calculate confidence score.
     */
    protected function calculateConfidenceScore(array $response, array $relevantDocs): float
    {
        if (empty($relevantDocs)) {
            return 0.0;
        }

        // Base score on document relevance
        $avgSimilarity = array_sum(array_column($relevantDocs, 'similarity')) / count($relevantDocs);
        $baseScore = $avgSimilarity * 100;

        // Adjust based on answer quality indicators
        $answer = $response['answer'];
        
        // Penalize if answer suggests contacting support
        if (preg_match('/contact.*support|speak.*agent|human.*help/i', $answer)) {
            $baseScore *= 0.6;
        }

        // Penalize if answer is too short
        if (strlen($answer) < 50) {
            $baseScore *= 0.7;
        }

        // Boost if answer includes specific details
        if (preg_match('/\d+|specific|exactly|precisely/i', $answer)) {
            $baseScore *= 1.1;
        }

        return min(100.0, max(0.0, $baseScore));
    }

    /**
     * Prepare citations from documents.
     */
    protected function prepareCitations(array $scoredDocs): array
    {
        $citations = [];

        foreach ($scoredDocs as $item) {
            $doc = $item['document'];
            $citations[] = [
                'document_id' => $doc->id,
                'title' => $doc->title,
                'document_type' => $doc->document_type,
                'relevance_score' => round($item['similarity'] * 100, 1),
            ];
        }

        return $citations;
    }

    /**
     * Handle case when no relevant documents found.
     */
    protected function handleNoRelevantDocuments(string $question): array
    {
        return [
            'answer' => "I apologize, but I don't have specific information about that in my knowledge base. Let me connect you with a human agent who can better assist you.",
            'citations' => [],
            'confidence_score' => 0.0,
            'requires_escalation' => true,
            'metadata' => [
                'reason' => 'no_relevant_documents',
            ],
        ];
    }

    /**
     * Handle errors.
     */
    protected function handleError(Exception $e): array
    {
        return [
            'answer' => "I'm experiencing technical difficulties. Please let me connect you with a human agent.",
            'citations' => [],
            'confidence_score' => 0.0,
            'requires_escalation' => true,
            'metadata' => [
                'error' => $e->getMessage(),
            ],
        ];
    }

    /**
     * Index a document (generate and store embedding).
     */
    public function indexDocument(KnowledgeBaseDocument $document): void
    {
        try {
            $textToEmbed = $document->title . "\n\n" . $document->content;
            $embedding = $this->generateEmbedding($textToEmbed);

            $document->update([
                'vector_embedding' => $embedding,
                'last_indexed_at' => now(),
            ]);

            Log::info('Document indexed successfully', [
                'document_id' => $document->id,
                'title' => $document->title,
            ]);

        } catch (Exception $e) {
            Log::error('Document indexing failed', [
                'document_id' => $document->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Batch index multiple documents.
     */
    public function batchIndexDocuments(int $brandId): array
    {
        $documents = KnowledgeBaseDocument::where('brand_id', $brandId)
            ->active()
            ->get();

        $results = [
            'total' => $documents->count(),
            'indexed' => 0,
            'failed' => 0,
            'errors' => [],
        ];

        foreach ($documents as $document) {
            try {
                if ($document->needsReindexing()) {
                    $this->indexDocument($document);
                    $results['indexed']++;
                }
            } catch (Exception $e) {
                $results['failed']++;
                $results['errors'][] = [
                    'document_id' => $document->id,
                    'error' => $e->getMessage(),
                ];
            }
        }

        return $results;
    }
}