<?php

namespace App\Services;

use App\Models\RazorpaySetting;
use App\Models\Subscription;
use App\Models\SubscriptionTransaction;
use App\Models\SubscriptionInvoice;
use App\Models\Plan;
use App\Models\User;
use Razorpay\Api\Api;
use Illuminate\Support\Facades\Log;
use Exception;

class RazorpayService
{
    protected $api;
    protected $settings;

    public function __construct()
    {
        $this->settings = RazorpaySetting::getInstance();
        
        if ($this->settings && $this->settings->isConfigured()) {
            $credentials = $this->settings->getCredentials();
            $this->api = new Api($credentials['key_id'], $credentials['key_secret']);
        }
    }

    /**
     * Check if Razorpay is configured and active.
     */
    public function isActive(): bool
    {
        return $this->settings && $this->settings->isActive();
    }

    /**
     * Create a Razorpay order for subscription.
     */
    public function createOrder(Plan $plan, User $user): array
    {
        if (!$this->isActive()) {
            throw new Exception('Razorpay is not configured or active.');
        }

        try {
            $amount = $plan->price * 100; // Convert to paise
            
            $orderData = [
                'receipt' => 'order_' . time(),
                'amount' => $amount,
                'currency' => $plan->currency,
                'notes' => [
                    'plan_id' => $plan->id,
                    'plan_slug' => $plan->slug,
                    'user_id' => $user->id,
                    'user_email' => $user->email,
                ]
            ];

            $order = $this->api->order->create($orderData);

            return [
                'order_id' => $order->id,
                'amount' => $amount,
                'currency' => $plan->currency,
                'key_id' => $this->settings->key_id,
            ];
        } catch (Exception $e) {
            Log::error('Razorpay order creation failed: ' . $e->getMessage());
            throw new Exception('Failed to create payment order: ' . $e->getMessage());
        }
    }

    /**
     * Verify payment signature.
     */
    public function verifyPaymentSignature(array $attributes): bool
    {
        try {
            $this->api->utility->verifyPaymentSignature($attributes);
            return true;
        } catch (Exception $e) {
            Log::error('Razorpay signature verification failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Process successful payment and create/update subscription.
     */
    public function processSuccessfulPayment(
        User $user,
        Plan $plan,
        string $razorpayPaymentId,
        string $razorpayOrderId,
        string $razorpaySignature
    ): Subscription {
        try {
            // Verify signature
            $isValid = $this->verifyPaymentSignature([
                'razorpay_payment_id' => $razorpayPaymentId,
                'razorpay_order_id' => $razorpayOrderId,
                'razorpay_signature' => $razorpaySignature,
            ]);

            if (!$isValid) {
                throw new Exception('Invalid payment signature');
            }

            // Fetch payment details from Razorpay
            $payment = $this->api->payment->fetch($razorpayPaymentId);

            // Create or update subscription
            $subscription = Subscription::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'tenant_id' => $user->tenant_id,
                ],
                [
                    'plan_id' => $plan->id,
                    'plan' => $plan->slug,
                    'status' => $plan->hasTrial() ? Subscription::STATUS_TRIALING : Subscription::STATUS_ACTIVE,
                    'trial_ends_at' => $plan->hasTrial() ? now()->addDays($plan->trial_days) : null,
                    'current_period_start' => now(),
                    'current_period_end' => now()->addMonth(),
                    'metadata' => [
                        'razorpay_payment_id' => $razorpayPaymentId,
                        'razorpay_order_id' => $razorpayOrderId,
                    ],
                ]
            );

            // Create transaction record
            $transaction = SubscriptionTransaction::create([
                'subscription_id' => $subscription->id,
                'user_id' => $user->id,
                'razorpay_payment_id' => $razorpayPaymentId,
                'razorpay_order_id' => $razorpayOrderId,
                'razorpay_signature' => $razorpaySignature,
                'type' => SubscriptionTransaction::TYPE_PAYMENT,
                'amount' => $plan->price,
                'currency' => $plan->currency,
                'status' => SubscriptionTransaction::STATUS_SUCCESS,
                'description' => "Payment for {$plan->name} plan",
                'paid_at' => now(),
                'metadata' => [
                    'payment_method' => $payment->method ?? null,
                    'card_last4' => $payment->card['last4'] ?? null,
                    'card_network' => $payment->card['network'] ?? null,
                ],
            ]);

            // Create invoice
            $this->createInvoice($subscription, $transaction, $plan);

            return $subscription;
        } catch (Exception $e) {
            Log::error('Payment processing failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Create invoice for a transaction.
     */
    protected function createInvoice(
        Subscription $subscription,
        SubscriptionTransaction $transaction,
        Plan $plan
    ): SubscriptionInvoice {
        $subtotal = $plan->price;
        $tax = 0; // Calculate tax if needed (e.g., GST)
        $total = $subtotal + $tax;

        return SubscriptionInvoice::create([
            'subscription_id' => $subscription->id,
            'user_id' => $subscription->user_id,
            'transaction_id' => $transaction->id,
            'invoice_number' => SubscriptionInvoice::generateInvoiceNumber(),
            'subtotal' => $subtotal,
            'tax' => $tax,
            'total' => $total,
            'currency' => $plan->currency,
            'status' => SubscriptionInvoice::STATUS_PAID,
            'issued_at' => now(),
            'paid_at' => now(),
            'line_items' => [
                [
                    'description' => $plan->name . ' - ' . ucfirst($plan->interval) . ' subscription',
                    'quantity' => 1,
                    'unit_price' => $plan->price,
                    'amount' => $plan->price,
                ]
            ],
        ]);
    }

    /**
     * Handle failed payment.
     */
    public function handleFailedPayment(
        User $user,
        Plan $plan,
        string $razorpayOrderId,
        string $errorMessage
    ): void {
        try {
            $subscription = $user->subscription;

            if ($subscription) {
                SubscriptionTransaction::create([
                    'subscription_id' => $subscription->id,
                    'user_id' => $user->id,
                    'razorpay_order_id' => $razorpayOrderId,
                    'type' => SubscriptionTransaction::TYPE_PAYMENT,
                    'amount' => $plan->price,
                    'currency' => $plan->currency,
                    'status' => SubscriptionTransaction::STATUS_FAILED,
                    'description' => "Failed payment for {$plan->name} plan",
                    'metadata' => [
                        'error' => $errorMessage,
                    ],
                ]);
            }
        } catch (Exception $e) {
            Log::error('Failed payment handling error: ' . $e->getMessage());
        }
    }

    /**
     * Create a Razorpay customer.
     */
    public function createCustomer(User $user): ?string
    {
        if (!$this->isActive()) {
            return null;
        }

        try {
            $customer = $this->api->customer->create([
                'name' => $user->name,
                'email' => $user->email,
                'contact' => $user->phone ?? '',
                'notes' => [
                    'user_id' => $user->id,
                ]
            ]);

            return $customer->id;
        } catch (Exception $e) {
            Log::error('Razorpay customer creation failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create a recurring subscription.
     */
    public function createRecurringSubscription(User $user, Plan $plan): ?array
    {
        if (!$this->isActive()) {
            throw new Exception('Razorpay is not configured or active.');
        }

        try {
            // Create customer if not exists
            $customerId = $user->subscription->razorpay_customer_id ?? $this->createCustomer($user);

            if (!$customerId) {
                throw new Exception('Failed to create Razorpay customer');
            }

            // Create subscription plan in Razorpay if needed
            $razorpayPlan = $this->createOrGetRazorpayPlan($plan);

            // Create subscription
            $subscription = $this->api->subscription->create([
                'plan_id' => $razorpayPlan['id'],
                'customer_id' => $customerId,
                'quantity' => 1,
                'total_count' => 12, // 12 months
                'start_at' => now()->addDays($plan->trial_days)->timestamp,
                'notes' => [
                    'user_id' => $user->id,
                    'plan_id' => $plan->id,
                ]
            ]);

            return [
                'subscription_id' => $subscription->id,
                'customer_id' => $customerId,
                'status' => $subscription->status,
            ];
        } catch (Exception $e) {
            Log::error('Razorpay recurring subscription creation failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Create or get Razorpay plan.
     */
    protected function createOrGetRazorpayPlan(Plan $plan): array
    {
        try {
            // Try to fetch existing plan
            $planId = 'plan_' . $plan->slug;
            
            try {
                $razorpayPlan = $this->api->plan->fetch($planId);
                return ['id' => $razorpayPlan->id];
            } catch (Exception $e) {
                // Plan doesn't exist, create it
            }

            // Create new plan
            $razorpayPlan = $this->api->plan->create([
                'period' => $plan->interval === 'monthly' ? 'monthly' : 'yearly',
                'interval' => 1,
                'item' => [
                    'name' => $plan->name,
                    'description' => $plan->description,
                    'amount' => $plan->price * 100, // Convert to paise
                    'currency' => $plan->currency,
                ],
                'notes' => [
                    'plan_id' => $plan->id,
                    'plan_slug' => $plan->slug,
                ]
            ]);

            return ['id' => $razorpayPlan->id];
        } catch (Exception $e) {
            Log::error('Razorpay plan creation failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Cancel a recurring subscription.
     */
    public function cancelRecurringSubscription(string $razorpaySubscriptionId): bool
    {
        if (!$this->isActive()) {
            return false;
        }

        try {
            $this->api->subscription->fetch($razorpaySubscriptionId)->cancel();
            return true;
        } catch (Exception $e) {
            Log::error('Razorpay subscription cancellation failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Process refund.
     */
    public function processRefund(string $paymentId, float $amount): bool
    {
        if (!$this->isActive()) {
            return false;
        }

        try {
            $this->api->payment->fetch($paymentId)->refund([
                'amount' => $amount * 100, // Convert to paise
            ]);
            return true;
        } catch (Exception $e) {
            Log::error('Razorpay refund failed: ' . $e->getMessage());
            return false;
        }
    }
}