<?php

namespace App\Services;

use App\Models\Brand;
use App\Models\ReputationScore;
use App\Models\ReputationDriver;
use App\Models\ReputationRecommendation;
use App\Models\ReputationAuditLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;

class ReputationScoreService
{
    protected AIService $aiService;

    public function __construct(AIService $aiService)
    {
        $this->aiService = $aiService;
    }

    /**
     * Calculate reputation score for a brand.
     */
    public function calculateScore(Brand $brand, string $timeframe = '30', array $feedData = []): ReputationScore
    {
        $periodEnd = now();
        $periodStart = $this->getPeriodStart($timeframe);

        // Get previous score for comparison
        $previousScore = ReputationScore::latestForBrand($brand->id)
            ->byTimeframe($timeframe)
            ->first();

        // Create new score record
        $reputationScore = ReputationScore::create([
            'brand_id' => $brand->id,
            'timeframe' => $timeframe,
            'period_start' => $periodStart,
            'period_end' => $periodEnd,
            'status' => 'calculating',
            'previous_score' => $previousScore?->score,
            'ai_job_id' => Str::uuid(),
        ]);

        try {
            // Aggregate data from various sources
            $aggregatedData = $this->aggregateData($brand, $periodStart, $periodEnd, $feedData);

            // Calculate sub-scores
            $sentimentScore = $this->calculateSentimentScore($aggregatedData);
            $responsivenessScore = $this->calculateResponsivenessScore($aggregatedData);
            $trustScore = $this->calculateTrustScore($aggregatedData);
            $complianceScore = $this->calculateComplianceScore($aggregatedData);
            $growthScore = $this->calculateGrowthScore($aggregatedData, $previousScore);

            // Calculate overall score (weighted average)
            $overallScore = $this->calculateOverallScore([
                'sentiment' => $sentimentScore,
                'responsiveness' => $responsivenessScore,
                'trust' => $trustScore,
                'compliance' => $complianceScore,
                'growth' => $growthScore,
            ]);

            // Calculate score change
            $scoreChange = $previousScore ? $overallScore - $previousScore->score : 0;
            $scoreChangePercentage = $previousScore && $previousScore->score > 0
                ? ($scoreChange / $previousScore->score) * 100
                : 0;

            // Generate AI explanation
            $aiExplanation = $this->generateAIExplanation($brand, $aggregatedData, [
                'overall' => $overallScore,
                'sentiment' => $sentimentScore,
                'responsiveness' => $responsivenessScore,
                'trust' => $trustScore,
                'compliance' => $complianceScore,
                'growth' => $growthScore,
            ], $scoreChange);

            // Check for alerts
            $alertTriggered = $scoreChangePercentage <= -10;
            $alertReason = $alertTriggered
                ? "Score dropped by {$scoreChangePercentage}% in the last {$timeframe} days"
                : null;

            // Update reputation score
            $reputationScore->update([
                'score' => $overallScore,
                'sentiment_score' => $sentimentScore,
                'responsiveness_score' => $responsivenessScore,
                'trust_score' => $trustScore,
                'compliance_score' => $complianceScore,
                'growth_score' => $growthScore,
                'score_change' => $scoreChange,
                'score_change_percentage' => $scoreChangePercentage,
                'ai_explanation' => $aiExplanation,
                'calculation_metadata' => $aggregatedData,
                'status' => 'completed',
                'alert_triggered' => $alertTriggered,
                'alert_reason' => $alertReason,
            ]);

            // Create drivers
            $this->createDrivers($reputationScore, $aggregatedData);

            // Generate recommendations
            $this->generateRecommendations($reputationScore, $aggregatedData);

            // Log the calculation
            $this->logAudit($brand, $reputationScore, 'score_calculated', 'Reputation score calculated successfully');

            // Trigger alert if needed
            if ($alertTriggered) {
                $this->triggerAlert($reputationScore);
            }

            return $reputationScore->fresh();
        } catch (\Exception $e) {
            Log::error('Reputation score calculation failed', [
                'brand_id' => $brand->id,
                'error' => $e->getMessage(),
            ]);

            $reputationScore->update([
                'status' => 'failed',
                'calculation_metadata' => ['error' => $e->getMessage()],
            ]);

            throw $e;
        }
    }

    /**
     * Recalculate score with manual overrides.
     */
    public function recalculateScore(ReputationScore $reputationScore): ReputationScore
    {
        $brand = $reputationScore->brand;
        
        // Get resolved drivers
        $resolvedDrivers = $reputationScore->drivers()->resolved()->get();
        
        // Adjust scores based on resolved issues
        $adjustments = $this->calculateResolvedAdjustments($resolvedDrivers);
        
        $newScore = min(100, $reputationScore->score + $adjustments['overall']);
        $scoreChange = $newScore - $reputationScore->score;
        
        $reputationScore->update([
            'score' => $newScore,
            'sentiment_score' => min(100, $reputationScore->sentiment_score + $adjustments['sentiment']),
            'compliance_score' => min(100, $reputationScore->compliance_score + $adjustments['compliance']),
            'score_change' => $reputationScore->score_change + $scoreChange,
        ]);

        $this->logAudit(
            $brand,
            $reputationScore,
            'score_recalculated',
            'Score recalculated after manual overrides',
            ['adjustments' => $adjustments]
        );

        return $reputationScore->fresh();
    }

    /**
     * Aggregate data from various sources.
     */
    protected function aggregateData(Brand $brand, Carbon $start, Carbon $end, array $feedData): array
    {
        return [
            'social_engagement' => $feedData['social_engagement'] ?? $this->getSocialEngagement($brand, $start, $end),
            'reviews' => $feedData['reviews'] ?? $this->getReviews($brand, $start, $end),
            'press_mentions' => $feedData['press_mentions'] ?? $this->getPressMentions($brand, $start, $end),
            'influencer_mentions' => $feedData['influencer_mentions'] ?? $this->getInfluencerMentions($brand, $start, $end),
            'complaints' => $feedData['complaints'] ?? $this->getComplaints($brand, $start, $end),
            'response_times' => $feedData['response_times'] ?? $this->getResponseTimes($brand, $start, $end),
            'legal_regulatory' => $feedData['legal_regulatory'] ?? $this->getLegalRegulatory($brand, $start, $end),
        ];
    }

    /**
     * Calculate sentiment score (0-100).
     */
    protected function calculateSentimentScore(array $data): float
    {
        $weights = [
            'social_engagement' => 0.3,
            'reviews' => 0.4,
            'press_mentions' => 0.2,
            'influencer_mentions' => 0.1,
        ];

        $score = 0;
        foreach ($weights as $key => $weight) {
            if (isset($data[$key]['sentiment'])) {
                // Convert -1 to 1 scale to 0-100
                $normalizedSentiment = ($data[$key]['sentiment'] + 1) * 50;
                $score += $normalizedSentiment * $weight;
            }
        }

        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Calculate responsiveness score (0-100).
     */
    protected function calculateResponsivenessScore(array $data): float
    {
        $responseTimes = $data['response_times'] ?? [];
        
        if (empty($responseTimes['average_minutes'])) {
            return 50; // Neutral score if no data
        }

        $avgMinutes = $responseTimes['average_minutes'];
        
        // Score based on response time (faster = better)
        // < 1 hour = 100, 1-4 hours = 80-100, 4-24 hours = 50-80, > 24 hours = 0-50
        if ($avgMinutes < 60) {
            $score = 100;
        } elseif ($avgMinutes < 240) {
            $score = 100 - (($avgMinutes - 60) / 180) * 20;
        } elseif ($avgMinutes < 1440) {
            $score = 80 - (($avgMinutes - 240) / 1200) * 30;
        } else {
            $score = max(0, 50 - (($avgMinutes - 1440) / 1440) * 50);
        }

        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Calculate trust score (0-100).
     */
    protected function calculateTrustScore(array $data): float
    {
        $pressMentions = $data['press_mentions'] ?? [];
        $influencerMentions = $data['influencer_mentions'] ?? [];
        
        $verifiedCount = ($pressMentions['verified_count'] ?? 0) + ($influencerMentions['verified_count'] ?? 0);
        $totalCount = ($pressMentions['total_count'] ?? 0) + ($influencerMentions['total_count'] ?? 0);
        
        if ($totalCount === 0) {
            return 50; // Neutral if no mentions
        }

        $verificationRatio = $verifiedCount / $totalCount;
        $reputabilityScore = ($pressMentions['reputable_outlets'] ?? 0) * 5; // Each reputable outlet adds 5 points
        
        $score = ($verificationRatio * 70) + min(30, $reputabilityScore);
        
        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Calculate compliance score (0-100).
     */
    protected function calculateComplianceScore(array $data): float
    {
        $legalRegulatory = $data['legal_regulatory'] ?? [];
        $complaints = $data['complaints'] ?? [];
        
        $flaggedContent = $legalRegulatory['flagged_content'] ?? 0;
        $legalIssues = $legalRegulatory['legal_issues'] ?? 0;
        $complaintVolume = $complaints['volume'] ?? 0;
        
        // Start at 100 and deduct for issues
        $score = 100;
        $score -= $flaggedContent * 5; // Each flagged content -5 points
        $score -= $legalIssues * 15; // Each legal issue -15 points
        $score -= min(30, $complaintVolume * 2); // Complaints impact (max -30)
        
        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Calculate growth score (0-100).
     */
    protected function calculateGrowthScore(array $data, ?ReputationScore $previousScore): float
    {
        if (!$previousScore) {
            return 50; // Neutral if no previous data
        }

        $currentEngagement = $data['social_engagement']['total_engagement'] ?? 0;
        $previousEngagement = $previousScore->calculation_metadata['social_engagement']['total_engagement'] ?? 0;
        
        if ($previousEngagement === 0) {
            return 50;
        }

        $growthRate = (($currentEngagement - $previousEngagement) / $previousEngagement) * 100;
        
        // Convert growth rate to 0-100 score
        // -50% = 0, 0% = 50, +50% = 100
        $score = 50 + ($growthRate);
        
        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Calculate overall score from sub-scores.
     */
    protected function calculateOverallScore(array $subScores): float
    {
        $weights = [
            'sentiment' => 0.30,
            'responsiveness' => 0.20,
            'trust' => 0.20,
            'compliance' => 0.20,
            'growth' => 0.10,
        ];

        $score = 0;
        foreach ($weights as $key => $weight) {
            $score += ($subScores[$key] ?? 50) * $weight;
        }

        return round(min(100, max(0, $score)), 2);
    }

    /**
     * Generate AI explanation for score.
     */
    protected function generateAIExplanation(Brand $brand, array $data, array $scores, float $change): string
    {
        $prompt = "Generate a concise 3-bullet explanation for a brand reputation score.\n\n";
        $prompt .= "Brand: {$brand->name}\n";
        $prompt .= "Overall Score: {$scores['overall']}/100\n";
        $prompt .= "Score Change: " . ($change >= 0 ? '+' : '') . "{$change}\n\n";
        $prompt .= "Sub-scores:\n";
        foreach ($scores as $key => $value) {
            if ($key !== 'overall') {
                $prompt .= "- " . ucfirst($key) . ": {$value}/100\n";
            }
        }
        $prompt .= "\nProvide 3 bullet points explaining the key drivers of this score.";

        try {
            $response = $this->aiService->generateText($prompt, [
                'max_tokens' => 200,
                'temperature' => 0.7,
            ]);

            return $response;
        } catch (\Exception $e) {
            Log::warning('AI explanation generation failed', ['error' => $e->getMessage()]);
            
            // Fallback explanation
            return $this->generateFallbackExplanation($scores, $change);
        }
    }

    /**
     * Generate fallback explanation without AI.
     */
    protected function generateFallbackExplanation(array $scores, float $change): string
    {
        $bullets = [];
        
        // Analyze score change
        if ($change > 5) {
            $bullets[] = "• Overall reputation improved by " . round($change, 1) . " points, indicating positive brand momentum";
        } elseif ($change < -5) {
            $bullets[] = "• Overall reputation declined by " . abs(round($change, 1)) . " points, requiring attention to key issues";
        } else {
            $bullets[] = "• Overall reputation remains stable with minor fluctuations";
        }

        // Find strongest and weakest areas
        unset($scores['overall']);
        arsort($scores);
        $strongest = array_key_first($scores);
        $weakest = array_key_last($scores);

        $bullets[] = "• Strongest area: " . ucfirst($strongest) . " (" . $scores[$strongest] . "/100) demonstrates brand strength";
        $bullets[] = "• Area for improvement: " . ucfirst($weakest) . " (" . $scores[$weakest] . "/100) needs strategic focus";

        return implode("\n", $bullets);
    }

    /**
     * Create reputation drivers from aggregated data.
     */
    protected function createDrivers(ReputationScore $reputationScore, array $data): void
    {
        $drivers = [];

        // Process social engagement
        foreach ($data['social_engagement']['posts'] ?? [] as $post) {
            $drivers[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'type' => $post['sentiment'] > 0 ? 'positive' : ($post['sentiment'] < 0 ? 'negative' : 'neutral'),
                'category' => 'social_engagement',
                'source_type' => 'post',
                'source_id' => $post['id'] ?? null,
                'source_url' => $post['url'] ?? null,
                'source_platform' => $post['platform'] ?? null,
                'source_content' => $post['content'] ?? null,
                'impact_score' => abs($post['sentiment'] ?? 0) * ($post['engagement'] ?? 1) / 100,
                'engagement_count' => $post['engagement'] ?? 0,
                'sentiment_value' => $post['sentiment'] ?? 0,
                'occurred_at' => $post['created_at'] ?? now(),
                'metadata' => $post,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Process reviews
        foreach ($data['reviews']['items'] ?? [] as $review) {
            $drivers[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'type' => $review['rating'] >= 4 ? 'positive' : ($review['rating'] <= 2 ? 'negative' : 'neutral'),
                'category' => 'review',
                'source_type' => 'review',
                'source_id' => $review['id'] ?? null,
                'source_url' => $review['url'] ?? null,
                'source_platform' => $review['platform'] ?? null,
                'source_content' => $review['content'] ?? null,
                'impact_score' => ($review['rating'] - 3) * 2, // -4 to +4
                'engagement_count' => $review['helpful_count'] ?? 0,
                'sentiment_value' => ($review['rating'] - 3) / 2, // -1 to +1
                'occurred_at' => $review['created_at'] ?? now(),
                'metadata' => $review,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Process complaints
        foreach ($data['complaints']['items'] ?? [] as $complaint) {
            $drivers[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'type' => 'negative',
                'category' => 'complaint',
                'source_type' => 'complaint',
                'source_id' => $complaint['id'] ?? null,
                'source_url' => $complaint['url'] ?? null,
                'source_platform' => $complaint['platform'] ?? null,
                'source_content' => $complaint['content'] ?? null,
                'impact_score' => -($complaint['severity'] ?? 3),
                'engagement_count' => $complaint['views'] ?? 0,
                'sentiment_value' => -0.8,
                'occurred_at' => $complaint['created_at'] ?? now(),
                'metadata' => $complaint,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Batch insert drivers (limit to top 100 by impact)
        if (!empty($drivers)) {
            usort($drivers, fn($a, $b) => abs($b['impact_score']) <=> abs($a['impact_score']));
            $topDrivers = array_slice($drivers, 0, 100);
            
            DB::table('reputation_drivers')->insert($topDrivers);
        }
    }

    /**
     * Generate recommendations based on score analysis.
     */
    protected function generateRecommendations(ReputationScore $reputationScore, array $data): void
    {
        $recommendations = [];

        // Sentiment improvement
        if ($reputationScore->sentiment_score < 70) {
            $recommendations[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'title' => 'Improve Social Sentiment',
                'description' => 'Focus on positive engagement and address negative feedback promptly to improve overall sentiment.',
                'priority' => $reputationScore->sentiment_score < 50 ? 'high' : 'medium',
                'category' => 'sentiment_improvement',
                'estimated_impact' => 15,
                'effort_level' => 'medium',
                'estimated_days' => 30,
                'ai_reasoning' => 'Low sentiment score indicates need for proactive community engagement',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Response time
        if ($reputationScore->responsiveness_score < 70) {
            $recommendations[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'title' => 'Reduce Response Time',
                'description' => 'Implement faster response protocols for customer inquiries and comments to improve responsiveness.',
                'priority' => 'high',
                'category' => 'response_time',
                'estimated_impact' => 12,
                'effort_level' => 'low',
                'estimated_days' => 14,
                'ai_reasoning' => 'Slow response times negatively impact customer satisfaction',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Trust building
        if ($reputationScore->trust_score < 70) {
            $recommendations[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'title' => 'Build Trust Signals',
                'description' => 'Seek verified mentions from reputable outlets and influencers to strengthen brand credibility.',
                'priority' => 'medium',
                'category' => 'trust_building',
                'estimated_impact' => 10,
                'effort_level' => 'high',
                'estimated_days' => 60,
                'ai_reasoning' => 'Limited verified mentions reduce perceived trustworthiness',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Compliance
        if ($reputationScore->compliance_score < 80) {
            $recommendations[] = [
                'reputation_score_id' => $reputationScore->id,
                'brand_id' => $reputationScore->brand_id,
                'title' => 'Address Compliance Issues',
                'description' => 'Review and resolve flagged content and legal/regulatory concerns immediately.',
                'priority' => 'high',
                'category' => 'compliance',
                'estimated_impact' => 20,
                'effort_level' => 'high',
                'estimated_days' => 7,
                'ai_reasoning' => 'Compliance issues pose significant reputation risk',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Insert recommendations
        if (!empty($recommendations)) {
            DB::table('reputation_recommendations')->insert($recommendations);
        }
    }

    /**
     * Calculate adjustments from resolved drivers.
     */
    protected function calculateResolvedAdjustments(iterable $resolvedDrivers): array
    {
        $adjustments = [
            'overall' => 0,
            'sentiment' => 0,
            'compliance' => 0,
        ];

        foreach ($resolvedDrivers as $driver) {
            if ($driver->type === 'negative') {
                $impact = abs($driver->impact_score);
                $adjustments['overall'] += $impact * 0.5; // 50% recovery
                
                if ($driver->category === 'complaint' || $driver->category === 'social_engagement') {
                    $adjustments['sentiment'] += $impact * 0.7;
                }
                
                if ($driver->category === 'legal_regulatory') {
                    $adjustments['compliance'] += $impact * 0.8;
                }
            }
        }

        return $adjustments;
    }

    /**
     * Log audit trail.
     */
    protected function logAudit(Brand $brand, ReputationScore $reputationScore, string $action, string $description, array $metadata = []): void
    {
        ReputationAuditLog::create([
            'brand_id' => $brand->id,
            'reputation_score_id' => $reputationScore->id,
            'action' => $action,
            'description' => $description,
            'metadata' => $metadata,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }

    /**
     * Trigger alert for significant score drop.
     */
    protected function triggerAlert(ReputationScore $reputationScore): void
    {
        // Log alert
        $this->logAudit(
            $reputationScore->brand,
            $reputationScore,
            'alert_triggered',
            $reputationScore->alert_reason
        );

        // TODO: Send notifications to configured admins
        // This would integrate with your notification system
        Log::info('Reputation alert triggered', [
            'brand_id' => $reputationScore->brand_id,
            'score' => $reputationScore->score,
            'change' => $reputationScore->score_change_percentage,
        ]);
    }

    /**
     * Get period start date based on timeframe.
     */
    protected function getPeriodStart(string $timeframe): Carbon
    {
        return now()->subDays((int)$timeframe);
    }

    // Placeholder methods for data aggregation
    // These would integrate with your actual data sources

    protected function getSocialEngagement(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with social media APIs
        return [
            'total_engagement' => 0,
            'sentiment' => 0,
            'posts' => [],
        ];
    }

    protected function getReviews(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with review platforms
        return [
            'average_rating' => 0,
            'total_reviews' => 0,
            'items' => [],
        ];
    }

    protected function getPressMentions(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with media monitoring
        return [
            'total_count' => 0,
            'verified_count' => 0,
            'reputable_outlets' => 0,
        ];
    }

    protected function getInfluencerMentions(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with influencer tracking
        return [
            'total_count' => 0,
            'verified_count' => 0,
        ];
    }

    protected function getComplaints(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with support system
        return [
            'volume' => 0,
            'items' => [],
        ];
    }

    protected function getResponseTimes(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with support/social response tracking
        return [
            'average_minutes' => 0,
            'total_responses' => 0,
        ];
    }

    protected function getLegalRegulatory(Brand $brand, Carbon $start, Carbon $end): array
    {
        // TODO: Integrate with compliance monitoring
        return [
            'flagged_content' => 0,
            'legal_issues' => 0,
        ];
    }
}