<?php

namespace App\Services\SocialProviders;

use App\Models\SocialAccount;
use Illuminate\Support\Facades\Config;
use Laravel\Socialite\Facades\Socialite;

class InstagramProvider extends BaseProvider
{
    /**
     * Configure the provider.
     *
     * @return void
     */
    protected function configure()
    {
        $this->slug = 'instagram';
        $this->name = 'Instagram';
        
        $this->endpoints = [
            'authorize' => 'https://api.instagram.com/oauth/authorize',
            'token' => 'https://api.instagram.com/oauth/access_token',
            'graph' => 'https://graph.instagram.com',
        ];
        
        $this->scopes = [
            'user_profile',
            'user_media',
        ];
        
        $this->clientId = Config::get('services.instagram.client_id');
        $this->clientSecret = Config::get('services.instagram.client_secret');
        $this->redirectUri = route('creator.connections.callback', ['provider' => 'instagram']);
    }

    /**
     * Redirect the user to the provider's authentication page.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function redirectToProvider($request)
    {
        // For Instagram, we'll use Socialite as it's well-supported
        return Socialite::driver('instagram')->redirect();
    }

    /**
     * Handle the callback from the provider.
     *
     * @param \Illuminate\Http\Request $request
     * @return array
     */
    public function handleCallback($request)
    {
        // For Instagram, we'll use Socialite as it's well-supported
        $user = Socialite::driver('instagram')->user();
        
        // Get long-lived token
        $longLivedToken = $this->getLongLivedToken($user->token);
        
        return [
            'token' => [
                'access_token' => $longLivedToken,
                'token_type' => 'bearer',
                'expires_in' => 5184000, // 60 days
            ],
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'nickname' => $user->nickname,
                'email' => $user->email,
                'avatar' => $user->avatar,
            ],
        ];
    }

    /**
     * Get a long-lived token from a short-lived token.
     *
     * @param string $shortLivedToken
     * @return string
     */
    protected function getLongLivedToken($shortLivedToken)
    {
        $url = 'https://graph.instagram.com/access_token';
        $params = [
            'grant_type' => 'ig_exchange_token',
            'client_secret' => $this->clientSecret,
            'access_token' => $shortLivedToken,
        ];
        
        $response = $this->httpClient->get($url, [
            'query' => $params,
        ]);
        
        $data = json_decode($response->getBody(), true);
        return $data['access_token'];
    }

    /**
     * Get user info using the access token.
     *
     * @param string $token
     * @return array
     */
    protected function getUserInfo($token)
    {
        $url = $this->endpoints['graph'] . '/me?fields=id,username,account_type,media_count&access_token=' . urlencode($token);
        
        $response = $this->httpClient->get($url);
        return json_decode($response->getBody(), true);
    }

    /**
     * Fetch account details from the provider.
     *
     * @param SocialAccount $account
     * @return array
     */
    public function fetchAccountDetails(SocialAccount $account): array
    {
        // For demo accounts, return fake data
        if ($account->is_demo) {
            return [
                'id' => $account->provider_user_id,
                'username' => 'demo_instagram',
                'followers' => 25000,
                'following' => 1200,
                'media_count' => 350,
                'profile_picture' => 'https://placehold.co/150x150',
            ];
        }

        try {
            // Get user info
            $url = $this->endpoints['graph'] . '/me?fields=id,username,account_type,media_count&access_token=' . urlencode($account->getAccessToken());
            $response = $this->httpClient->get($url);
            $userData = json_decode($response->getBody(), true);
            
            // For business accounts, we can get more detailed info
            if ($userData['account_type'] === 'BUSINESS') {
                // Get insights for business accounts
                $insightsUrl = $this->endpoints['graph'] . '/me?fields=followers_count,follows_count&access_token=' . urlencode($account->getAccessToken());
                $insightsResponse = $this->httpClient->get($insightsUrl);
                $insightsData = json_decode($insightsResponse->getBody(), true);
                
                return [
                    'id' => $userData['id'],
                    'username' => $userData['username'],
                    'account_type' => $userData['account_type'],
                    'media_count' => $userData['media_count'],
                    'followers' => $insightsData['followers_count'] ?? 0,
                    'following' => $insightsData['follows_count'] ?? 0,
                ];
            } else {
                // For personal accounts, we can't get follower count directly
                return [
                    'id' => $userData['id'],
                    'username' => $userData['username'],
                    'account_type' => $userData['account_type'],
                    'media_count' => $userData['media_count'],
                ];
            }
        } catch (\Exception $e) {
            // Log the error and return empty array
            $this->logRequestException($e, 'Failed to fetch Instagram account details', $account);
            return [];
        }
    }

    /**
     * Fetch recent posts from the provider.
     *
     * @param SocialAccount $account
     * @param int $limit
     * @return array
     */
    public function fetchRecentPosts(SocialAccount $account, int $limit = 20): array
    {
        // For demo accounts, return fake data
        if ($account->is_demo) {
            $posts = [];
            for ($i = 1; $i <= min($limit, 10); $i++) {
                $posts[] = [
                    'id' => 'demo_post_' . $i,
                    'caption' => 'This is a demo Instagram post #' . $i,
                    'media_type' => $i % 3 === 0 ? 'VIDEO' : 'IMAGE',
                    'media_url' => 'https://placehold.co/600x600',
                    'permalink' => 'https://www.instagram.com/p/demo_post_' . $i,
                    'timestamp' => now()->subDays($i)->toIso8601String(),
                    'likes' => rand(100, 2000),
                    'comments' => rand(10, 200),
                ];
            }
            return $posts;
        }

        try {
            // Get recent media
            $url = $this->endpoints['graph'] . '/me/media?fields=id,caption,media_type,media_url,permalink,timestamp&limit=' . $limit . '&access_token=' . urlencode($account->getAccessToken());
            $response = $this->httpClient->get($url);
            $data = json_decode($response->getBody(), true);
            
            $posts = [];
            foreach ($data['data'] as $item) {
                $posts[] = [
                    'id' => $item['id'],
                    'caption' => $item['caption'] ?? '',
                    'media_type' => $item['media_type'],
                    'media_url' => $item['media_url'],
                    'permalink' => $item['permalink'],
                    'timestamp' => $item['timestamp'],
                ];
            }
            
            return $posts;
        } catch (\Exception $e) {
            // Log the error and return empty array
            $this->logRequestException($e, 'Failed to fetch Instagram posts', $account);
            return [];
        }
    }
    
    /**
     * Post content to Instagram.
     *
     * @param ScheduledPost $scheduledPost
     * @param SocialAccount $account
     * @return array
     */
    public function postContent($scheduledPost, SocialAccount $account): array
    {
        try {
            // For demo accounts, return success
            if ($account->is_demo) {
                return [
                    'status' => 'success',
                    'message' => 'Demo post successful',
                    'remote_post_id' => 'demo_' . uniqid(),
                ];
            }
            
            // Check if we have media to post
            $mediaRefs = $scheduledPost->getDecryptedMediaRefs();
            
            if (empty($mediaRefs)) {
                // Text-only post (Instagram doesn't support this directly, so we'll create an image with text)
                // For now, we'll return an error
                return [
                    'status' => 'failed',
                    'message' => 'Instagram requires media for posts',
                ];
            }
            
            // For simplicity, we'll only post the first media item
            $media = $mediaRefs[0];
            
            // Create media container
            $containerUrl = $this->endpoints['graph'] . '/me/media?';
            $containerParams = [
                'image_url' => $media['url'],
                'caption' => $scheduledPost->caption,
                'access_token' => $account->getAccessToken(),
            ];
            
            $response = $this->httpClient->post($containerUrl, [
                'form_params' => $containerParams,
            ]);
            
            $containerData = json_decode($response->getBody(), true);
            
            if (!isset($containerData['id'])) {
                return [
                    'status' => 'failed',
                    'message' => 'Failed to create media container',
                ];
            }
            
            // Publish the container
            $publishUrl = $this->endpoints['graph'] . '/me/media_publish?';
            $publishParams = [
                'creation_id' => $containerData['id'],
                'access_token' => $account->getAccessToken(),
            ];
            
            $response = $this->httpClient->post($publishUrl, [
                'form_params' => $publishParams,
            ]);
            
            $publishData = json_decode($response->getBody(), true);
            
            if (!isset($publishData['id'])) {
                return [
                    'status' => 'failed',
                    'message' => 'Failed to publish media',
                ];
            }
            
            return [
                'status' => 'success',
                'message' => 'Post published successfully',
                'remote_post_id' => $publishData['id'],
            ];
        } catch (\Exception $e) {
            $this->logRequestException($e, 'Failed to post content to Instagram', $account);
            return [
                'status' => 'failed',
                'message' => 'Failed to post content: ' . $e->getMessage(),
            ];
        }
    }
}