# AI Business Market - Marketing Frontend Documentation

## Overview

This document provides comprehensive information about the marketing frontend implementation for AI Business Market. The frontend is built with Laravel Blade templates, Tailwind CSS, Alpine.js, and follows a component-driven architecture.

## Table of Contents

1. [Architecture](#architecture)
2. [Pages](#pages)
3. [Components](#components)
4. [Design System](#design-system)
5. [Analytics](#analytics)
6. [Performance](#performance)
7. [SEO](#seo)
8. [Accessibility](#accessibility)
9. [Testing](#testing)
10. [Deployment](#deployment)

---

## Architecture

### Technology Stack

- **Backend Framework**: Laravel 10+
- **Template Engine**: Blade
- **CSS Framework**: Tailwind CSS 3.x
- **JavaScript**: Alpine.js 3.x
- **Build Tool**: Vite
- **Analytics**: Google Analytics 4, Facebook Pixel
- **Payment**: Razorpay Checkout

### Directory Structure

```
resources/
├── css/
│   ├── app.css                 # Main CSS file
│   └── design-tokens.css       # Design system variables
├── js/
│   ├── app.js                  # Main JavaScript entry
│   └── analytics.js            # Analytics tracking helper
└── views/
    ├── layouts/
    │   └── marketing.blade.php # Marketing layout
    ├── marketing/
    │   ├── home.blade.php      # Landing page
    │   ├── pricing.blade.php   # Pricing page
    │   ├── demo.blade.php      # Product demo
    │   ├── help.blade.php      # Help center
    │   └── onboarding/
    │       ├── brand.blade.php
    │       ├── creator.blade.php
    │       └── agency.blade.php
    ├── subscription/
    │   ├── checkout.blade.php  # Checkout page
    │   ├── success.blade.php   # Success page
    │   └── manage.blade.php    # Manage subscription
    └── components/
        └── marketing/
            ├── hero.blade.php
            ├── pricing-card.blade.php
            ├── role-cards.blade.php
            ├── feature-grid.blade.php
            ├── testimonials.blade.php
            ├── faq.blade.php
            ├── trust-badges.blade.php
            ├── sticky-cta.blade.php
            ├── checkout-modal.blade.php
            ├── cookie-consent.blade.php
            └── search-dropdown.blade.php
```

---

## Pages

### 1. Landing Page (`/`)

**Route**: `marketing.home`  
**File**: `resources/views/marketing/home.blade.php`

**Features**:
- Dynamic hero with role-based messaging (Brand/Creator/Agency)
- Role selection cards
- Feature highlights grid
- Social proof and testimonials
- How it works section
- FAQ accordion
- Sticky bottom CTA (mobile)

**Analytics Events**:
- `page_view`
- `role_selected`
- `cta_click`

### 2. Pricing Page (`/pricing`)

**Route**: `marketing.pricing`  
**File**: `resources/views/marketing/pricing.blade.php`

**Features**:
- Monthly/Yearly toggle with savings badge
- Role filter (Brand/Creator/Agency/All)
- Plan comparison cards
- Feature comparison table (desktop) / Accordion (mobile)
- Trust badges
- FAQ section

**Analytics Events**:
- `pricing_view`
- `plan_click`
- `interval_toggle`

### 3. Demo Page (`/demo`)

**Route**: `marketing.demo`  
**File**: `resources/views/marketing/demo.blade.php`

**Features**:
- Video player with thumbnail
- Interactive feature tabs
- Screenshot galleries
- CTA sections

**Analytics Events**:
- `demo_started`
- `video_play`
- `tab_clicked`

### 4. Help Center (`/help`)

**Route**: `marketing.help`  
**File**: `resources/views/marketing/help.blade.php`

**Features**:
- Search functionality
- Category quick links
- Popular articles
- FAQ section
- Contact support CTA

**Analytics Events**:
- `search`
- `article_click`
- `contact_support`

### 5. Onboarding Pages

**Routes**: 
- `marketing.onboarding.brand`
- `marketing.onboarding.creator`
- `marketing.onboarding.agency`

**Features**:
- Multi-step wizard
- Progress indicator
- Role-specific flows
- Social account connection
- Goal setting
- Team invitations

**Analytics Events**:
- `onboarding_started`
- `onboarding_step_completed`
- `onboarding_completed`

### 6. Checkout Page (`/subscription/checkout/{plan}`)

**Route**: `subscription.checkout`  
**File**: `resources/views/subscription/checkout.blade.php`

**Features**:
- Plan summary
- Razorpay integration
- Trial information
- Terms acceptance

**Analytics Events**:
- `checkout_initiated`
- `payment_success`
- `payment_failed`

### 7. Manage Subscription (`/subscription/manage`)

**Route**: `subscription.manage`  
**File**: `resources/views/subscription/manage.blade.php`

**Features**:
- Current plan details
- Usage limits display
- Upgrade/downgrade options
- Invoice history
- Cancellation flow

**Analytics Events**:
- `subscription_viewed`
- `upgrade_initiated`
- `subscription_cancelled`

---

## Components

### Reusable Blade Components

All components are located in `resources/views/components/marketing/`

#### 1. Hero Component

```blade
<x-marketing.hero :selectedRole="$selectedRole" />
```

**Props**:
- `selectedRole` (string): 'brand', 'creator', or 'agency'

**Features**:
- Role-based headlines
- Dynamic CTAs
- Animated background
- Product mockup
- Social proof numbers

#### 2. Pricing Card

```blade
<x-marketing.pricing-card 
    :plan="$plan" 
    :featured="true"
    interval="monthly"
    :selectedRole="$role" 
/>
```

**Props**:
- `plan` (object): Plan model instance
- `featured` (boolean): Highlight as recommended
- `interval` (string): 'monthly' or 'yearly'
- `selectedRole` (string): Filter by role

#### 3. Checkout Modal

```blade
<x-marketing.checkout-modal :plan="$plan">
    Start Trial
</x-marketing.checkout-modal>
```

**Props**:
- `plan` (object): Plan model instance

**Features**:
- Pre-checkout confirmation
- Plan summary
- Terms display
- Razorpay trigger

#### 4. Cookie Consent

```blade
<x-marketing.cookie-consent />
```

**Features**:
- GDPR compliant
- LocalStorage persistence
- Google Consent Mode integration
- Accept/Decline options

#### 5. Search Dropdown

```blade
<x-marketing.search-dropdown />
```

**Features**:
- Typeahead search
- Keyboard navigation
- Result categorization
- No results state

#### 6. FAQ Component

```blade
<x-marketing.faq />
```

**Features**:
- Accordion interface
- Structured data (JSON-LD)
- Search functionality
- Category filtering

---

## Design System

### Design Tokens

All design tokens are defined in `resources/css/design-tokens.css`

#### Color Palette

```css
--color-primary-500: #3b82f6;    /* Blue */
--color-secondary-500: #a855f7;  /* Purple */
--color-accent-500: #d946ef;     /* Pink */
--color-success: #10b981;        /* Green */
--color-warning: #f59e0b;        /* Amber */
--color-error: #ef4444;          /* Red */
```

#### Typography Scale

```css
--text-xs: 0.75rem;    /* 12px */
--text-sm: 0.875rem;   /* 14px */
--text-base: 1rem;     /* 16px */
--text-lg: 1.125rem;   /* 18px */
--text-xl: 1.25rem;    /* 20px */
--text-2xl: 1.5rem;    /* 24px */
--text-3xl: 1.875rem;  /* 30px */
--text-4xl: 2.25rem;   /* 36px */
--text-5xl: 3rem;      /* 48px */
```

#### Spacing

```css
--space-1: 0.25rem;   /* 4px */
--space-2: 0.5rem;    /* 8px */
--space-4: 1rem;      /* 16px */
--space-6: 1.5rem;    /* 24px */
--space-8: 2rem;      /* 32px */
--space-12: 3rem;     /* 48px */
```

#### Breakpoints

```css
--breakpoint-sm: 640px;
--breakpoint-md: 768px;
--breakpoint-lg: 1024px;
--breakpoint-xl: 1280px;
```

### Component Patterns

#### Button Styles

```html
<!-- Primary Button -->
<button class="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition">
    Primary Action
</button>

<!-- Secondary Button -->
<button class="bg-white border-2 border-gray-300 text-gray-800 font-semibold py-3 px-6 rounded-lg hover:bg-gray-50 transition">
    Secondary Action
</button>
```

#### Card Styles

```html
<div class="bg-white rounded-2xl shadow-xl p-6">
    <!-- Card content -->
</div>
```

---

## Analytics

### Implementation

Analytics tracking is handled by `resources/js/analytics.js`

### Usage

```javascript
// Track custom event
window.tracker.track('event_name', {
    property1: 'value1',
    property2: 'value2'
});

// Track pricing view
window.tracker.pricingView('plan-id', 'brand');

// Track checkout
window.tracker.checkoutInitiated('plan-id', 'Pro Plan', 999);

// Track payment success
window.tracker.paymentSuccess('plan-id', 999, 'razorpay_id', 'user_id');
```

### Standard Events

| Event Name | Description | Properties |
|------------|-------------|------------|
| `page_view` | Page viewed | `page_path`, `page_title` |
| `pricing_view` | Pricing page viewed | `plan_id`, `role` |
| `plan_click` | Plan CTA clicked | `plan_id`, `plan_name`, `role` |
| `checkout_initiated` | Checkout started | `plan_id`, `amount`, `user_id` |
| `payment_success` | Payment completed | `plan_id`, `amount`, `razorpay_id` |
| `trial_started` | Trial activated | `plan_id`, `trial_days` |
| `upgrade_initiated` | Upgrade clicked | `from_plan`, `to_plan` |
| `contact_sales` | Contact sales clicked | `source` |
| `cta_click` | CTA button clicked | `cta_text`, `cta_location` |
| `video_play` | Video started | `video_title`, `video_url` |
| `search` | Search performed | `search_term`, `results_count` |

### UTM Tracking

UTM parameters are automatically captured and stored in session storage:

```javascript
// Get UTM parameters
const utm = window.tracker.getUTM();
// Returns: { utm_source, utm_medium, utm_campaign, utm_term, utm_content }
```

---

## Performance

### Targets

- **First Contentful Paint (FCP)**: < 1.5s (desktop)
- **Time to Interactive (TTI)**: < 3s (mobile)
- **Lighthouse Performance Score**: ≥ 80
- **Lighthouse Accessibility Score**: ≥ 90

### Optimization Strategies

1. **Critical CSS Inline**: Hero and header styles inlined
2. **Lazy Loading**: Below-fold images and components
3. **Code Splitting**: Separate bundles for marketing vs. app
4. **Image Optimization**: WebP/AVIF with fallbacks
5. **Preconnect**: External domains (fonts, Razorpay)
6. **Defer Non-Critical JS**: Analytics, chat widgets

### Image Guidelines

```html
<!-- Responsive images with WebP -->
<picture>
    <source srcset="image.webp" type="image/webp">
    <source srcset="image.jpg" type="image/jpeg">
    <img src="image.jpg" alt="Description" loading="lazy">
</picture>
```

---

## SEO

### Meta Tags

Each page includes:
- Title (50-60 characters)
- Description (150-160 characters)
- Canonical URL
- OpenGraph tags
- Twitter Card tags

### Structured Data

Implemented using JSON-LD:

```blade
@push('structured-data')
<script type="application/ld+json">
{
    "@context": "https://schema.org",
    "@type": "Product",
    "name": "AI Business Market",
    ...
}
</script>
@endpush
```

### Sitemap

Generate sitemap with:
```bash
php artisan sitemap:generate
```

---

## Accessibility

### WCAG AA Compliance

- ✅ Color contrast ratios ≥ 4.5:1
- ✅ Keyboard navigation
- ✅ Screen reader support
- ✅ ARIA labels and roles
- ✅ Focus indicators
- ✅ Skip to content link

### Testing Tools

- axe DevTools
- WAVE Browser Extension
- Lighthouse Accessibility Audit
- Screen Reader (NVDA/JAWS)

---

## Testing

### QA Checklist

#### Visual QA
- [ ] Pixel-perfect match to Figma designs
- [ ] Responsive on all breakpoints (mobile, tablet, desktop)
- [ ] Cross-browser compatibility (Chrome, Firefox, Safari, Edge)
- [ ] Dark mode support (if applicable)

#### Functional QA
- [ ] All links work correctly
- [ ] Forms submit successfully
- [ ] Pricing toggle updates prices
- [ ] Role selection filters plans
- [ ] Razorpay modal opens and processes payments
- [ ] Cookie consent saves preferences
- [ ] Search returns relevant results

#### Performance QA
- [ ] Lighthouse Performance ≥ 80
- [ ] Lighthouse Accessibility ≥ 90
- [ ] Images optimized and lazy-loaded
- [ ] No console errors

#### Analytics QA
- [ ] All events fire correctly
- [ ] UTM parameters captured
- [ ] Events visible in GA4 dashboard

---

## Deployment

### Build Process

```bash
# Install dependencies
npm install

# Build for production
npm run build

# Optimize images
npm run optimize-images
```

### Environment Variables

```env
# Analytics
GOOGLE_ANALYTICS_ID=G-XXXXXXXXXX
FACEBOOK_PIXEL_ID=XXXXXXXXXX

# Razorpay
RAZORPAY_KEY_ID=rzp_live_XXXXXXXXXX
RAZORPAY_KEY_SECRET=XXXXXXXXXX

# App
APP_URL=https://aibusinessmarket.com
```

### Cache Clearing

```bash
# Clear all caches
php artisan optimize:clear

# Rebuild caches
php artisan optimize
```

---

## Support

For questions or issues:
- **Email**: dev@aibusinessmarket.com
- **Slack**: #frontend-team
- **Documentation**: https://docs.aibusinessmarket.com

---

## Changelog

### v1.0.0 (2024-01-15)
- Initial marketing frontend implementation
- Landing, pricing, demo, help pages
- Checkout and subscription management
- Analytics tracking
- Cookie consent
- Role-based onboarding

---

**Last Updated**: 2024-01-15  
**Maintained By**: Frontend Team