@extends('layouts.app')

@section('title', 'Financial Insights Dashboard')

@section('content')
<div class="container mx-auto px-4 py-8">
    <!-- Header -->
    <div class="flex justify-between items-center mb-8">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Financial Insights</h1>
            <p class="text-gray-600 mt-2">Track ROI, budget usage, and revenue impact of your campaigns</p>
        </div>
        <div class="flex gap-3">
            <button onclick="showPeriodSelector()" class="btn btn-secondary">
                <i class="fas fa-calendar"></i> Change Period
            </button>
            <button onclick="generateReport()" class="btn btn-primary">
                <i class="fas fa-chart-line"></i> Generate Report
            </button>
        </div>
    </div>

    <!-- Period Display -->
    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
        <div class="flex items-center justify-between">
            <div class="flex items-center gap-2">
                <i class="fas fa-calendar-alt text-blue-600"></i>
                <span class="font-medium text-gray-700">Current Period:</span>
                <span class="text-gray-900">{{ $periodStart->format('M d, Y') }} - {{ $periodEnd->format('M d, Y') }}</span>
            </div>
            <span class="text-sm text-gray-600">{{ $periodStart->diffInDays($periodEnd) }} days</span>
        </div>
    </div>

    <!-- Overview Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <!-- Total Spend -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-sm font-medium text-gray-600">Total Spend</h3>
                <div class="p-2 bg-red-100 rounded-lg">
                    <i class="fas fa-dollar-sign text-red-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-900">${{ number_format($data['overview']['total_spend'], 2) }}</p>
            @if(isset($data['trends']['spend_trend']))
            <div class="mt-2 flex items-center text-sm">
                <i class="fas fa-arrow-{{ $data['trends']['spend_trend']['direction'] === 'up' ? 'up text-red-500' : 'down text-green-500' }} mr-1"></i>
                <span class="{{ $data['trends']['spend_trend']['direction'] === 'up' ? 'text-red-600' : 'text-green-600' }}">
                    {{ abs($data['trends']['spend_trend']['value']) }}% vs last period
                </span>
            </div>
            @endif
        </div>

        <!-- Total Revenue -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-sm font-medium text-gray-600">Total Revenue</h3>
                <div class="p-2 bg-green-100 rounded-lg">
                    <i class="fas fa-chart-line text-green-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-900">${{ number_format($data['overview']['total_revenue'], 2) }}</p>
            @if(isset($data['trends']['revenue_trend']))
            <div class="mt-2 flex items-center text-sm">
                <i class="fas fa-arrow-{{ $data['trends']['revenue_trend']['direction'] === 'up' ? 'up' : 'down' }} mr-1 text-{{ $data['trends']['revenue_trend']['direction'] === 'up' ? 'green' : 'red' }}-500"></i>
                <span class="text-{{ $data['trends']['revenue_trend']['direction'] === 'up' ? 'green' : 'red' }}-600">
                    {{ abs($data['trends']['revenue_trend']['value']) }}% vs last period
                </span>
            </div>
            @endif
        </div>

        <!-- Net Profit -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-sm font-medium text-gray-600">Net Profit</h3>
                <div class="p-2 bg-blue-100 rounded-lg">
                    <i class="fas fa-coins text-blue-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold {{ $data['overview']['net_profit'] >= 0 ? 'text-green-600' : 'text-red-600' }}">
                ${{ number_format($data['overview']['net_profit'], 2) }}
            </p>
            <div class="mt-2 text-sm text-gray-600">
                {{ $data['overview']['net_profit'] >= 0 ? 'Profitable' : 'Loss' }}
            </div>
        </div>

        <!-- ROI -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-sm font-medium text-gray-600">ROI</h3>
                <div class="p-2 bg-purple-100 rounded-lg">
                    <i class="fas fa-percentage text-purple-600"></i>
                </div>
            </div>
            <p class="text-3xl font-bold text-gray-900">{{ number_format($data['overview']['roi_percentage'], 2) }}%</p>
            <div class="mt-2">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                    @if($data['overview']['roi_status'] === 'excellent') bg-green-100 text-green-800
                    @elseif($data['overview']['roi_status'] === 'good') bg-blue-100 text-blue-800
                    @elseif($data['overview']['roi_status'] === 'positive') bg-yellow-100 text-yellow-800
                    @else bg-red-100 text-red-800
                    @endif">
                    {{ ucfirst($data['overview']['roi_status']) }}
                </span>
            </div>
        </div>
    </div>

    <!-- Charts Row -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- Spend vs Revenue Chart -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Spend vs Revenue Trend</h3>
            <canvas id="spendRevenueChart" height="300"></canvas>
        </div>

        <!-- ROI Trend Chart -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">ROI Trend</h3>
            <canvas id="roiTrendChart" height="300"></canvas>
        </div>
    </div>

    <!-- Channel Performance -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-8">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Channel Performance</h3>
        <canvas id="channelPerformanceChart" height="200"></canvas>
    </div>

    <!-- AI Insights -->
    @if(!empty($data['insights']))
    <div class="bg-gradient-to-r from-purple-50 to-blue-50 rounded-lg shadow-md p-6 mb-8">
        <div class="flex items-center gap-2 mb-4">
            <i class="fas fa-robot text-purple-600 text-xl"></i>
            <h3 class="text-lg font-semibold text-gray-900">AI-Powered Insights</h3>
            <span class="ml-auto text-sm text-gray-600">{{ count($data['insights']) }} insights</span>
        </div>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            @foreach($data['insights'] as $insight)
            <div class="bg-white rounded-lg p-4 border-l-4 
                @if($insight['type'] === 'opportunity') border-green-500
                @elseif($insight['type'] === 'warning') border-yellow-500
                @else border-blue-500
                @endif">
                <div class="flex items-start justify-between mb-2">
                    <h4 class="font-semibold text-gray-900">{{ $insight['title'] }}</h4>
                    <span class="px-2 py-1 text-xs rounded-full
                        @if($insight['priority'] === 'high') bg-red-100 text-red-800
                        @else bg-blue-100 text-blue-800
                        @endif">
                        {{ ucfirst($insight['priority']) }}
                    </span>
                </div>
                <p class="text-sm text-gray-600 mb-2">{{ $insight['description'] }}</p>
                <div class="flex items-center text-sm text-green-600">
                    <i class="fas fa-arrow-up mr-1"></i>
                    <span>{{ $insight['expected_impact'] }}</span>
                </div>
            </div>
            @endforeach
        </div>

        @if($data['recommendations'])
        <div class="mt-4 p-4 bg-white rounded-lg">
            <h4 class="font-semibold text-gray-900 mb-2">Recommendations</h4>
            <p class="text-sm text-gray-700 whitespace-pre-line">{{ $data['recommendations'] }}</p>
        </div>
        @endif
    </div>
    @endif

    <!-- Budget Scenarios -->
    @if(!empty($data['scenarios']))
    <div class="bg-white rounded-lg shadow-md p-6 mb-8">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-900">Budget Scenarios</h3>
            <button onclick="generateScenarios()" class="btn btn-sm btn-secondary">
                <i class="fas fa-plus"></i> Generate New
            </button>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            @foreach($data['scenarios'] as $scenario)
            <div class="border rounded-lg p-4 hover:shadow-lg transition-shadow cursor-pointer"
                 onclick="viewScenario({{ $scenario->id }})">
                <div class="flex items-center justify-between mb-3">
                    <h4 class="font-semibold text-gray-900">{{ $scenario->name }}</h4>
                    @if($scenario->is_applied)
                    <span class="px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">Active</span>
                    @endif
                </div>
                
                <div class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-600">Budget:</span>
                        <span class="font-medium">${{ number_format($scenario->total_budget, 2) }}</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Projected ROI:</span>
                        <span class="font-medium text-green-600">{{ number_format($scenario->projected_roi, 2) }}%</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Risk Level:</span>
                        <span class="font-medium">{{ ucfirst($scenario->getRiskLevel()) }}</span>
                    </div>
                </div>

                @if(!$scenario->is_applied)
                <button onclick="applyScenario({{ $scenario->id }}); event.stopPropagation();" 
                        class="mt-3 w-full btn btn-sm btn-primary">
                    Apply Scenario
                </button>
                @endif
            </div>
            @endforeach
        </div>
    </div>
    @endif

    <!-- Top Performing Campaigns -->
    @if(!empty($data['top_campaigns']))
    <div class="bg-white rounded-lg shadow-md p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">Top Performing Campaigns</h3>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Campaign</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Spend</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Revenue</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">ROI</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @foreach($data['top_campaigns'] as $campaign)
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900">{{ $campaign->campaign->name ?? 'N/A' }}</div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            ${{ number_format($campaign->total_spend, 2) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            ${{ number_format($campaign->total_revenue, 2) }}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="text-sm font-medium {{ $campaign->roi_percentage >= 0 ? 'text-green-600' : 'text-red-600' }}">
                                {{ number_format($campaign->roi_percentage, 2) }}%
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 py-1 text-xs rounded-full
                                @if($campaign->getRoiStatus() === 'positive') bg-green-100 text-green-800
                                @elseif($campaign->getRoiStatus() === 'breakeven') bg-yellow-100 text-yellow-800
                                @else bg-red-100 text-red-800
                                @endif">
                                {{ ucfirst($campaign->getRoiStatus()) }}
                            </span>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    @endif
</div>

<!-- Disclaimer -->
<div class="container mx-auto px-4 py-4">
    <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div class="flex items-start">
            <i class="fas fa-exclamation-triangle text-yellow-600 mt-1 mr-3"></i>
            <div class="text-sm text-gray-700">
                <strong>Disclaimer:</strong> AI-generated insights and projections are estimates based on historical data. 
                Actual results may vary. This information is not financial advice. Please consult with financial professionals 
                for investment decisions.
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
<script>
// Load chart data
document.addEventListener('DOMContentLoaded', function() {
    loadSpendRevenueChart();
    loadROITrendChart();
    loadChannelPerformanceChart();
});

function loadSpendRevenueChart() {
    fetch('/brand/financial-insights/spend-vs-revenue?start_date={{ $periodStart->format("Y-m-d") }}&end_date={{ $periodEnd->format("Y-m-d") }}')
        .then(response => response.json())
        .then(data => {
            const ctx = document.getElementById('spendRevenueChart').getContext('2d');
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: data.data.map(d => d.campaign),
                    datasets: [{
                        label: 'Spend',
                        data: data.data.map(d => d.spend),
                        backgroundColor: 'rgba(239, 68, 68, 0.5)',
                        borderColor: 'rgb(239, 68, 68)',
                        borderWidth: 1
                    }, {
                        label: 'Revenue',
                        data: data.data.map(d => d.revenue),
                        backgroundColor: 'rgba(34, 197, 94, 0.5)',
                        borderColor: 'rgb(34, 197, 94)',
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return '$' + value.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
        });
}

function loadROITrendChart() {
    fetch('/brand/financial-insights/roi-trends?start_date={{ $periodStart->format("Y-m-d") }}&end_date={{ $periodEnd->format("Y-m-d") }}&granularity=weekly')
        .then(response => response.json())
        .then(data => {
            const ctx = document.getElementById('roiTrendChart').getContext('2d');
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.data.map(d => d.period),
                    datasets: [{
                        label: 'ROI %',
                        data: data.data.map(d => d.roi),
                        borderColor: 'rgb(147, 51, 234)',
                        backgroundColor: 'rgba(147, 51, 234, 0.1)',
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            ticks: {
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        }
                    }
                }
            });
        });
}

function loadChannelPerformanceChart() {
    fetch('/brand/financial-insights/channel-performance?start_date={{ $periodStart->format("Y-m-d") }}&end_date={{ $periodEnd->format("Y-m-d") }}')
        .then(response => response.json())
        .then(data => {
            if (!data.success) return;
            
            const ctx = document.getElementById('channelPerformanceChart').getContext('2d');
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: data.data.map(d => d.channel),
                    datasets: [{
                        label: 'ROI %',
                        data: data.data.map(d => d.roi),
                        backgroundColor: data.data.map(d => 
                            d.roi > 20 ? 'rgba(34, 197, 94, 0.5)' :
                            d.roi > 0 ? 'rgba(59, 130, 246, 0.5)' :
                            'rgba(239, 68, 68, 0.5)'
                        ),
                        borderColor: data.data.map(d => 
                            d.roi > 20 ? 'rgb(34, 197, 94)' :
                            d.roi > 0 ? 'rgb(59, 130, 246)' :
                            'rgb(239, 68, 68)'
                        ),
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'y',
                    scales: {
                        x: {
                            ticks: {
                                callback: function(value) {
                                    return value + '%';
                                }
                            }
                        }
                    }
                }
            });
        });
}

function generateReport() {
    // Implementation for generating new report
    window.location.href = '/brand/financial-insights/generate-report';
}

function generateScenarios() {
    // Implementation for generating scenarios
    window.location.href = '/brand/financial-insights/scenarios/generate';
}

function viewScenario(id) {
    window.location.href = `/brand/financial-insights/scenarios/${id}`;
}

function applyScenario(id) {
    if (confirm('Are you sure you want to apply this budget scenario?')) {
        fetch(`/brand/financial-insights/scenarios/${id}/apply`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Scenario applied successfully!');
                location.reload();
            } else {
                alert('Failed to apply scenario');
            }
        });
    }
}

function showPeriodSelector() {
    // Implementation for period selector modal
    alert('Period selector modal - to be implemented');
}
</script>
@endpush
@endsection