@extends('layouts.brand')

@section('title', 'Reputation Score Dashboard')

@section('content')
<div class="container mx-auto px-4 py-8">
    <!-- Header -->
    <div class="flex justify-between items-center mb-8">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Reputation Score</h1>
            <p class="text-gray-600 mt-1">Monitor and manage your brand's public reputation</p>
        </div>
        <div class="flex gap-3">
            <select id="timeframe" onchange="window.location.href='?timeframe='+this.value" class="rounded-lg border-gray-300">
                <option value="7" {{ $timeframe == '7' ? 'selected' : '' }}>Last 7 Days</option>
                <option value="30" {{ $timeframe == '30' ? 'selected' : '' }}>Last 30 Days</option>
                <option value="90" {{ $timeframe == '90' ? 'selected' : '' }}>Last 90 Days</option>
            </select>
            
            <form action="{{ route('brand.reputation.calculate') }}" method="POST" class="inline">
                @csrf
                <input type="hidden" name="timeframe" value="{{ $timeframe }}">
                <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    <i class="fas fa-sync-alt mr-2"></i>Recalculate
                </button>
            </form>
        </div>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-6">{{ session('success') }}</div>
    @endif

    @if($latestScore)
        <!-- Overall Score Card -->
        <div class="bg-white rounded-lg shadow-lg p-8 mb-8">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-8">
                <div class="text-center">
                    <div class="text-6xl font-bold mb-2 {{ $latestScore->score >= 85 ? 'text-green-600' : ($latestScore->score >= 70 ? 'text-blue-600' : ($latestScore->score >= 50 ? 'text-yellow-600' : 'text-red-600')) }}">
                        {{ number_format($latestScore->score, 1) }}
                    </div>
                    <div class="text-gray-600 text-sm">Overall Score</div>
                    <div class="mt-2">
                        <span class="px-3 py-1 rounded-full text-sm font-semibold {{ $latestScore->score >= 85 ? 'bg-green-100 text-green-800' : ($latestScore->score >= 70 ? 'bg-blue-100 text-blue-800' : ($latestScore->score >= 50 ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800')) }}">
                            Grade {{ $latestScore->grade }}
                        </span>
                    </div>
                </div>

                <div class="text-center">
                    <div class="text-3xl font-bold mb-2 {{ $latestScore->score_change >= 0 ? 'text-green-600' : 'text-red-600' }}">
                        {{ $latestScore->score_change >= 0 ? '+' : '' }}{{ number_format($latestScore->score_change, 1) }}
                    </div>
                    <div class="text-gray-600 text-sm">Change from Previous</div>
                    <div class="mt-2 text-sm text-gray-500">
                        {{ number_format(abs($latestScore->score_change_percentage), 1) }}% {{ $latestScore->trend }}
                    </div>
                </div>

                <div>
                    <h3 class="font-semibold mb-3">AI Insights</h3>
                    <div class="text-sm text-gray-700 space-y-2">
                        @foreach($latestScore->ai_explanation_array as $insight)
                            <div class="flex items-start">
                                <i class="fas fa-check-circle text-blue-500 mt-1 mr-2"></i>
                                <span>{{ $insight }}</span>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>

        <!-- Sub-Scores -->
        <div class="grid grid-cols-1 md:grid-cols-5 gap-4 mb-8">
            @foreach([
                ['name' => 'Sentiment', 'score' => $latestScore->sentiment_score, 'icon' => 'fa-smile'],
                ['name' => 'Responsiveness', 'score' => $latestScore->responsiveness_score, 'icon' => 'fa-clock'],
                ['name' => 'Trust', 'score' => $latestScore->trust_score, 'icon' => 'fa-shield-alt'],
                ['name' => 'Compliance', 'score' => $latestScore->compliance_score, 'icon' => 'fa-check-circle'],
                ['name' => 'Growth', 'score' => $latestScore->growth_score, 'icon' => 'fa-chart-line']
            ] as $subscore)
                <div class="bg-white rounded-lg shadow p-6 text-center">
                    <i class="fas {{ $subscore['icon'] }} text-3xl text-gray-400 mb-3"></i>
                    <div class="text-2xl font-bold mb-1">{{ number_format($subscore['score'], 1) }}</div>
                    <div class="text-sm text-gray-600">{{ $subscore['name'] }}</div>
                </div>
            @endforeach
        </div>

        <!-- Drivers Section -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-8 mb-8">
            <!-- Positive Drivers -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold mb-4 flex items-center">
                    <i class="fas fa-arrow-up text-green-500 mr-2"></i>
                    Top Positive Drivers
                </h3>
                <div class="space-y-3">
                    @forelse($topPositiveDrivers as $driver)
                        <div class="border-l-4 border-green-500 pl-4 py-2">
                            <div class="flex justify-between items-start">
                                <div class="flex-1">
                                    <div class="font-medium text-sm">{{ ucfirst(str_replace('_', ' ', $driver->category)) }}</div>
                                    <div class="text-xs text-gray-600 mt-1">{{ $driver->excerpt }}</div>
                                    @if($driver->source_url)
                                        <a href="{{ $driver->source_url }}" target="_blank" class="text-xs text-blue-600 hover:underline">View Source</a>
                                    @endif
                                </div>
                                <span class="text-green-600 font-semibold ml-2">+{{ number_format($driver->impact_score, 1) }}</span>
                            </div>
                        </div>
                    @empty
                        <p class="text-gray-500 text-sm">No positive drivers found</p>
                    @endforelse
                </div>
            </div>

            <!-- Negative Drivers -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold mb-4 flex items-center">
                    <i class="fas fa-arrow-down text-red-500 mr-2"></i>
                    Top Negative Drivers
                </h3>
                <div class="space-y-3">
                    @forelse($topNegativeDrivers as $driver)
                        <div class="border-l-4 border-red-500 pl-4 py-2">
                            <div class="flex justify-between items-start">
                                <div class="flex-1">
                                    <div class="font-medium text-sm">{{ ucfirst(str_replace('_', ' ', $driver->category)) }}</div>
                                    <div class="text-xs text-gray-600 mt-1">{{ $driver->excerpt }}</div>
                                    <div class="mt-2 flex gap-2">
                                        @if($driver->source_url)
                                            <a href="{{ $driver->source_url }}" target="_blank" class="text-xs text-blue-600 hover:underline">View Source</a>
                                        @endif
                                        @if(!$driver->is_resolved)
                                            <button onclick="resolveDriver({{ $driver->id }})" class="text-xs text-green-600 hover:underline">Mark Resolved</button>
                                        @else
                                            <span class="text-xs text-green-600">✓ Resolved</span>
                                        @endif
                                    </div>
                                </div>
                                <span class="text-red-600 font-semibold ml-2">{{ number_format($driver->impact_score, 1) }}</span>
                            </div>
                        </div>
                    @empty
                        <p class="text-gray-500 text-sm">No negative drivers found</p>
                    @endforelse
                </div>
            </div>
        </div>

        <!-- Recommendations -->
        <div class="bg-white rounded-lg shadow p-6 mb-8">
            <h3 class="text-lg font-semibold mb-4">Recommended Actions</h3>
            <div class="space-y-4">
                @forelse($recommendations->take(5) as $rec)
                    <div class="border rounded-lg p-4 {{ $rec->priority === 'high' ? 'border-red-300 bg-red-50' : ($rec->priority === 'medium' ? 'border-yellow-300 bg-yellow-50' : 'border-gray-300') }}">
                        <div class="flex justify-between items-start">
                            <div class="flex-1">
                                <div class="flex items-center gap-2 mb-2">
                                    <span class="px-2 py-1 rounded text-xs font-semibold {{ $rec->priority === 'high' ? 'bg-red-200 text-red-800' : ($rec->priority === 'medium' ? 'bg-yellow-200 text-yellow-800' : 'bg-gray-200 text-gray-800') }}">
                                        {{ strtoupper($rec->priority) }}
                                    </span>
                                    <span class="text-sm font-semibold">{{ $rec->title }}</span>
                                </div>
                                <p class="text-sm text-gray-700 mb-2">{{ $rec->description }}</p>
                                <div class="flex gap-4 text-xs text-gray-600">
                                    <span><i class="fas fa-chart-line mr-1"></i>Impact: +{{ $rec->estimated_impact }}</span>
                                    <span><i class="fas fa-tasks mr-1"></i>Effort: {{ ucfirst($rec->effort_level) }}</span>
                                    @if($rec->estimated_days)
                                        <span><i class="fas fa-calendar mr-1"></i>{{ $rec->estimated_days }} days</span>
                                    @endif
                                </div>
                            </div>
                            <div class="ml-4">
                                <select onchange="updateRecommendation({{ $rec->id }}, this.value)" class="text-sm rounded border-gray-300">
                                    <option value="pending" {{ $rec->status === 'pending' ? 'selected' : '' }}>Pending</option>
                                    <option value="in_progress" {{ $rec->status === 'in_progress' ? 'selected' : '' }}>In Progress</option>
                                    <option value="completed" {{ $rec->status === 'completed' ? 'selected' : '' }}>Completed</option>
                                    <option value="dismissed" {{ $rec->status === 'dismissed' ? 'selected' : '' }}>Dismissed</option>
                                </select>
                            </div>
                        </div>
                    </div>
                @empty
                    <p class="text-gray-500">No recommendations at this time</p>
                @endforelse
            </div>
        </div>

        <!-- Score History Chart -->
        <div class="bg-white rounded-lg shadow p-6">
            <h3 class="text-lg font-semibold mb-4">Score History</h3>
            <canvas id="scoreChart" height="80"></canvas>
        </div>
    @else
        <div class="bg-white rounded-lg shadow p-12 text-center">
            <i class="fas fa-chart-line text-6xl text-gray-300 mb-4"></i>
            <h3 class="text-xl font-semibold mb-2">No Reputation Score Yet</h3>
            <p class="text-gray-600 mb-6">Calculate your first reputation score to get started</p>
            <form action="{{ route('brand.reputation.calculate') }}" method="POST">
                @csrf
                <input type="hidden" name="timeframe" value="30">
                <button type="submit" class="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    Calculate Reputation Score
                </button>
            </form>
        </div>
    @endif
</div>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
<script>
@if($latestScore && $historicalScores->count() > 0)
    const ctx = document.getElementById('scoreChart').getContext('2d');
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: {!! json_encode($historicalScores->pluck('created_at')->map(fn($d) => $d->format('M d'))) !!},
            datasets: [{
                label: 'Overall Score',
                data: {!! json_encode($historicalScores->pluck('score')) !!},
                borderColor: 'rgb(59, 130, 246)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: true,
            scales: {
                y: { beginAtZero: true, max: 100 }
            }
        }
    });
@endif

function resolveDriver(driverId) {
    const note = prompt('Enter resolution note:');
    if (note) {
        fetch(`/brand/reputation/drivers/${driverId}/resolve`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ resolution_note: note })
        }).then(() => location.reload());
    }
}

function updateRecommendation(recId, status) {
    fetch(`/brand/reputation/recommendations/${recId}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({ status: status })
    }).then(() => location.reload());
}
</script>
@endpush
@endsection