@extends('adminlte::page')

@section('title', 'Scheduled Posts')

@section('content_header')
    <h1>Scheduled Posts</h1>
@stop

@section('content')
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Posts Calendar</h3>
                <div class="card-tools">
                    <a href="{{ route('brand.scheduled-posts.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Schedule New Post
                    </a>
                </div>
            </div>
            <div class="card-body">
                <div id="scheduled-posts-calendar"></div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h3 class="card-title">Scheduled Posts</h3>
                <div class="card-tools">
                    <form method="GET" action="{{ route('brand.scheduled-posts.index') }}" class="form-inline">
                        <div class="form-group mr-2">
                            <label for="status-filter" class="mr-2">Status:</label>
                            <select name="status" id="status-filter" class="form-control">
                                <option value="">All</option>
                                <option value="draft" {{ request('status') === 'draft' ? 'selected' : '' }}>Draft</option>
                                <option value="pending_approval" {{ request('status') === 'pending_approval' ? 'selected' : '' }}>Pending Approval</option>
                                <option value="scheduled" {{ request('status') === 'scheduled' ? 'selected' : '' }}>Scheduled</option>
                                <option value="publishing" {{ request('status') === 'publishing' ? 'selected' : '' }}>Publishing</option>
                                <option value="posted" {{ request('status') === 'posted' ? 'selected' : '' }}>Posted</option>
                                <option value="failed" {{ request('status') === 'failed' ? 'selected' : '' }}>Failed</option>
                                <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                                <option value="denied" {{ request('status') === 'denied' ? 'selected' : '' }}>Denied</option>
                            </select>
                        </div>
                        <div class="form-group mr-2">
                            <label for="campaign-filter" class="mr-2">Campaign:</label>
                            <select name="campaign_id" id="campaign-filter" class="form-control">
                                <option value="">All</option>
                                @foreach($campaigns as $campaign)
                                    <option value="{{ $campaign->id }}" {{ request('campaign_id') == $campaign->id ? 'selected' : '' }}>
                                        {{ $campaign->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                        <button type="submit" class="btn btn-default">Filter</button>
                    </form>
                </div>
            </div>
            <div class="card-body">
                @if($scheduledPosts->count() > 0)
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Social Account</th>
                                    <th>Caption</th>
                                    <th>Status</th>
                                    <th>Publish Time</th>
                                    <th>Timezone</th>
                                    <th>Campaign</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($scheduledPosts as $post)
                                    <tr>
                                        <td>{{ $post->id }}</td>
                                        <td>{{ $post->socialAccount->provider ?? 'N/A' }}</td>
                                        <td>{{ Str::limit($post->caption, 50) }}</td>
                                        <td>
                                            @if($post->status === 'draft')
                                                <span class="badge badge-secondary">Draft</span>
                                            @elseif($post->status === 'pending_approval')
                                                <span class="badge badge-warning">Pending Approval</span>
                                            @elseif($post->status === 'scheduled')
                                                <span class="badge badge-info">Scheduled</span>
                                            @elseif($post->status === 'publishing')
                                                <span class="badge badge-primary">Publishing</span>
                                            @elseif($post->status === 'posted')
                                                <span class="badge badge-success">Posted</span>
                                            @elseif($post->status === 'failed')
                                                <span class="badge badge-danger">Failed</span>
                                            @elseif($post->status === 'cancelled')
                                                <span class="badge badge-dark">Cancelled</span>
                                            @elseif($post->status === 'denied')
                                                <span class="badge badge-secondary">Denied</span>
                                            @endif
                                        </td>
                                        <td>{{ $post->publish_time ? $post->publish_time->format('Y-m-d H:i') : 'N/A' }}</td>
                                        <td>{{ $post->timezone ?? 'N/A' }}</td>
                                        <td>{{ $post->campaign->name ?? 'N/A' }}</td>
                                        <td>{{ $post->created_at->diffForHumans() }}</td>
                                        <td>
                                            <a href="{{ route('brand.scheduled-posts.show', $post) }}" class="btn btn-sm btn-info">
                                                <i class="fas fa-eye"></i> View
                                            </a>
                                            @if(in_array($post->status, ['scheduled', 'draft', 'pending_approval']))
                                                <form action="{{ route('brand.scheduled-posts.publish-now', $post) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-success">
                                                        <i class="fas fa-bolt"></i> Publish Now
                                                    </button>
                                                </form>
                                            @endif
                                            @if(in_array($post->status, ['scheduled', 'draft']))
                                                <a href="{{ route('brand.scheduled-posts.edit', $post) }}" class="btn btn-sm btn-warning">
                                                    <i class="fas fa-edit"></i> Edit
                                                </a>
                                            @endif
                                            @if(in_array($post->status, ['scheduled', 'draft', 'pending_approval']))
                                                <form action="{{ route('brand.scheduled-posts.cancel', $post) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to cancel this scheduled post?')">
                                                        <i class="fas fa-times"></i> Cancel
                                                    </button>
                                                </form>
                                            @endif
                                            @if($post->status === 'failed')
                                                <form action="{{ route('brand.scheduled-posts.retry', $post) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-sm btn-warning">
                                                        <i class="fas fa-redo"></i> Retry
                                                    </button>
                                                </form>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    <div class="d-flex justify-content-center">
                        {{ $scheduledPosts->links() }}
                    </div>
                @else
                    <p>No scheduled posts found.</p>
                @endif
            </div>
        </div>
    </div>
</div>
@stop

@section('css')
    <link rel="stylesheet" href="/css/admin_custom.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.css">
@stop

@section('js')
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.10.1/main.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var calendarEl = document.getElementById('scheduled-posts-calendar');
            if (calendarEl) {
                var calendar = new FullCalendar.Calendar(calendarEl, {
                    initialView: 'dayGridMonth',
                    headerToolbar: {
                        left: 'prev,next today',
                        center: 'title',
                        right: 'dayGridMonth,timeGridWeek,timeGridDay'
                    },
                    events: [
                        @foreach($scheduledPosts as $post)
                            @if($post->publish_time)
                                {
                                    title: '{{ Str::limit($post->caption, 30) }}',
                                    start: '{{ $post->publish_time->format('Y-m-d H:i:s') }}',
                                    classNames: ['status-{{ $post->status }}']
                                },
                            @endif
                        @endforeach
                    ]
                });
                calendar.render();
            }
        });
    </script>
@stop