<?php

namespace Tests\Feature;

use App\Models\Brand;
use App\Models\User;
use App\Jobs\EnqueueAiTaskJob;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Queue;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class AiTaskEnqueueTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;
    protected $brand;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->exists()) {
            Role::create(['name' => 'brand_admin']);
        }
        
        if (!Role::where('name', 'creator')->exists()) {
            Role::create(['name' => 'creator']);
        }

        // Create a brand
        $this->brand = Brand::factory()->create([
            'name' => 'Test Brand',
        ]);

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
            'brand_id' => $this->brand->id,
        ]);
        $this->brandAdmin->assignRole('brand_admin');
    }

    public function test_ai_task_is_enqueued_successfully()
    {
        // Fake the queue
        Queue::fake();

        // Act as the brand admin
        $response = $this->actingAs($this->brandAdmin)
            ->post(route('brand.ai-tasks.generate-proposal'));

        // Assert the response status
        $response->assertStatus(302);
        $response->assertRedirect(route('brand.dashboard'));
        $response->assertSessionHas('success');

        // Assert that the job was pushed to the queue
        Queue::assertPushed(EnqueueAiTaskJob::class, 1);

        // Assert that a row exists in the database with status "pending"
        $this->assertDatabaseHas('ai_tasks', [
            'type' => 'generate_proposal',
            'brand_id' => $this->brand->id,
            'status' => 'pending',
        ]);
    }

    public function test_non_brand_user_cannot_enqueue_ai_task()
    {
        // Create a creator user
        $creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $creator->assignRole('creator');

        // Fake the queue
        Queue::fake();

        // Act as the creator (not a brand admin)
        $response = $this->actingAs($creator)
            ->post(route('brand.ai-tasks.generate-proposal'));

        // Assert the response status (should be 403 Forbidden)
        $response->assertStatus(403);

        // Assert that the job was not pushed to the queue
        Queue::assertNotPushed(EnqueueAiTaskJob::class);

        // Assert that no row exists in the database
        $this->assertDatabaseMissing('ai_tasks', [
            'type' => 'generate_proposal',
        ]);
    }
}