<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Brand;
use App\Models\Tenant;
use App\Models\CompetitorBrand;
use App\Models\CompetitorPost;
use App\Models\CompetitorInsight;
use App\Models\CompetitorAlert;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class CompetitorInsightsTest extends TestCase
{
    use RefreshDatabase;

    protected User $brandAdmin;
    protected Brand $brand;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        // Create tenant
        $this->tenant = Tenant::factory()->create();

        // Create brand
        $this->brand = Brand::factory()->create([
            'tenant_id' => $this->tenant->id,
        ]);

        // Create brand admin user
        $this->brandAdmin = User::factory()->create([
            'tenant_id' => $this->tenant->id,
            'brand_id' => $this->brand->id,
        ]);

        // Create and assign brand_admin role
        $role = Role::firstOrCreate(['name' => 'brand_admin']);
        $this->brandAdmin->assignRole($role);
    }

    /** @test */
    public function brand_admin_can_access_competitor_insights_dashboard()
    {
        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.competitor-insights.index'));

        $response->assertStatus(200);
    }

    /** @test */
    public function non_brand_admin_cannot_access_competitor_insights()
    {
        $user = User::factory()->create([
            'tenant_id' => $this->tenant->id,
        ]);

        $response = $this->actingAs($user)
            ->get(route('brand.competitor-insights.index'));

        $response->assertStatus(403);
    }

    /** @test */
    public function brand_admin_can_create_competitor()
    {
        $competitorData = [
            'name' => 'Test Competitor',
            'description' => 'A test competitor',
            'industry' => 'Technology',
            'social_handles' => [
                'instagram' => '@test_competitor',
                'facebook' => 'test_competitor',
            ],
        ];

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.competitor-insights.competitors.store'), $competitorData);

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
                'message' => 'Competitor added successfully',
            ]);

        $this->assertDatabaseHas('competitor_brands', [
            'brand_id' => $this->brand->id,
            'name' => 'Test Competitor',
        ]);
    }

    /** @test */
    public function brand_admin_can_update_competitor()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $updateData = [
            'name' => 'Updated Competitor Name',
            'is_active' => false,
        ];

        $response = $this->actingAs($this->brandAdmin)
            ->putJson(route('brand.competitor-insights.competitors.update', $competitor), $updateData);

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
            ]);

        $this->assertDatabaseHas('competitor_brands', [
            'id' => $competitor->id,
            'name' => 'Updated Competitor Name',
            'is_active' => false,
        ]);
    }

    /** @test */
    public function brand_admin_can_delete_competitor()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->deleteJson(route('brand.competitor-insights.competitors.destroy', $competitor));

        $response->assertStatus(200)
            ->assertJson([
                'success' => true,
            ]);

        $this->assertDatabaseMissing('competitor_brands', [
            'id' => $competitor->id,
        ]);
    }

    /** @test */
    public function brand_admin_cannot_access_other_brands_competitors()
    {
        $otherBrand = Brand::factory()->create([
            'tenant_id' => $this->tenant->id,
        ]);

        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $otherBrand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->get(route('brand.competitor-insights.show', $competitor));

        $response->assertStatus(403);
    }

    /** @test */
    public function competitor_posts_are_stored_correctly()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $post = CompetitorPost::create([
            'competitor_brand_id' => $competitor->id,
            'platform' => 'instagram',
            'platform_post_id' => 'test_post_123',
            'post_type' => 'reel',
            'caption' => 'Test post caption',
            'metrics' => [
                'likes' => 1000,
                'comments' => 50,
                'shares' => 20,
                'views' => 10000,
            ],
            'published_at' => now(),
        ]);

        $this->assertDatabaseHas('competitor_posts', [
            'competitor_brand_id' => $competitor->id,
            'platform_post_id' => 'test_post_123',
        ]);

        $this->assertEquals(1000, $post->getLikesCount());
        $this->assertEquals(50, $post->getCommentsCount());
    }

    /** @test */
    public function competitor_insights_can_be_created()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $insight = CompetitorInsight::create([
            'brand_id' => $this->brand->id,
            'competitor_brand_id' => $competitor->id,
            'tenant_id' => $this->tenant->id,
            'start_date' => now()->subDays(30),
            'end_date' => now(),
            'competitor_metrics' => [
                'total_posts' => 20,
                'avg_engagement_rate' => 5.5,
            ],
            'brand_metrics' => [
                'total_posts' => 15,
                'avg_engagement_rate' => 4.2,
            ],
            'status' => 'completed',
        ]);

        $this->assertDatabaseHas('competitor_insights', [
            'competitor_brand_id' => $competitor->id,
            'status' => 'completed',
        ]);

        $this->assertTrue($insight->isCompleted());
    }

    /** @test */
    public function competitor_alerts_can_be_created()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $alert = CompetitorAlert::create([
            'brand_id' => $this->brand->id,
            'competitor_brand_id' => $competitor->id,
            'tenant_id' => $this->tenant->id,
            'alert_type' => 'spike',
            'metric' => 'engagement_rate',
            'severity' => 'high',
            'title' => 'Engagement Rate Spike',
            'description' => 'Competitor has seen a 50% increase in engagement',
            'current_value' => 7.5,
            'threshold_value' => 5.0,
            'change_percentage' => 50.0,
        ]);

        $this->assertDatabaseHas('competitor_alerts', [
            'competitor_brand_id' => $competitor->id,
            'alert_type' => 'spike',
        ]);

        $this->assertTrue($alert->isHighSeverity());
        $this->assertFalse($alert->is_read);
    }

    /** @test */
    public function alerts_can_be_marked_as_read()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $alert = CompetitorAlert::factory()->create([
            'brand_id' => $this->brand->id,
            'competitor_brand_id' => $competitor->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.competitor-insights.alerts.read', $alert));

        $response->assertStatus(200);

        $alert->refresh();
        $this->assertTrue($alert->is_read);
        $this->assertNotNull($alert->read_at);
    }

    /** @test */
    public function alerts_can_be_dismissed()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $alert = CompetitorAlert::factory()->create([
            'brand_id' => $this->brand->id,
            'competitor_brand_id' => $competitor->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $response = $this->actingAs($this->brandAdmin)
            ->postJson(route('brand.competitor-insights.alerts.dismiss', $alert));

        $response->assertStatus(200);

        $alert->refresh();
        $this->assertTrue($alert->is_dismissed);
        $this->assertNotNull($alert->dismissed_at);
    }

    /** @test */
    public function engagement_rate_is_calculated_correctly()
    {
        $competitor = CompetitorBrand::factory()->create([
            'brand_id' => $this->brand->id,
            'tenant_id' => $this->tenant->id,
        ]);

        $post = CompetitorPost::create([
            'competitor_brand_id' => $competitor->id,
            'platform' => 'instagram',
            'platform_post_id' => 'test_post_456',
            'metrics' => [
                'likes' => 100,
                'comments' => 10,
                'shares' => 5,
                'saves' => 5,
                'views' => 1000,
            ],
            'published_at' => now(),
        ]);

        $engagementRate = $post->calculateEngagementRate();
        
        // (100 + 10 + 5 + 5) / 1000 * 100 = 12%
        $this->assertEquals(12.0, $engagementRate);
    }
}