<?php

namespace Tests\Feature;

use App\Models\User;
use Database\Seeders\SocialDemoSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CreatorUiSmokeTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that the creator dashboard loads correctly with demo data.
     *
     * @return void
     */
    public function test_creator_dashboard_loads_with_demo_data()
    {
        // Seed demo data
        $this->seed(SocialDemoSeeder::class);

        // Get the demo creator user
        $user = User::where('email', 'creator@demo.local')->first();

        // Ensure the user exists
        $this->assertNotNull($user, 'Demo creator user not found');

        // Login as the creator user
        $response = $this->actingAs($user);

        // Visit the dashboard
        $response = $this->get(route('creator.dashboard'));

        // Assert successful response
        $response->assertStatus(200);

        // Assert that KPI numbers are present
        $response->assertSee('Total Followers/Subscribers');
        $response->assertSee('Total Posts/Videos');
        $response->assertSee('Total Views');
        $response->assertSee('Engagement Rate');

        // Assert that connection status section is present
        $response->assertSee('Connection Status');

        // Assert that recent posts section is present
        $response->assertSee('Recent Posts/Videos');
    }

    /**
     * Test that the creator connections page loads correctly.
     *
     * @return void
     */
    public function test_creator_connections_page_loads()
    {
        // Seed demo data
        $this->seed(SocialDemoSeeder::class);

        // Get the demo creator user
        $user = User::where('email', 'creator1@demo.local')->first();

        // Ensure the user exists
        $this->assertNotNull($user, 'Demo creator user not found');

        // Login as the creator user
        $response = $this->actingAs($user);

        // Visit the connections page
        $response = $this->get(route('creator.connections.index'));

        // Assert successful response
        $response->assertStatus(200);

        // Assert that connected accounts section is present
        $response->assertSee('Connected Accounts');

        // Assert that connect buttons are present
        $response->assertSee('Connect New Account');
        $response->assertSee('Add Manual Token');
    }

    /**
     * Test that the stats API endpoint returns valid data.
     *
     * @return void
     */
    public function test_stats_api_endpoint_returns_valid_data()
    {
        // Seed demo data
        $this->seed(SocialDemoSeeder::class);

        // Get the demo creator user
        $user = User::where('email', 'creator1@demo.local')->first();

        // Ensure the user exists
        $this->assertNotNull($user, 'Demo creator user not found');

        // Login as the creator user
        $response = $this->actingAs($user);

        // Call the stats API endpoint
        $response = $this->get(route('creator.dashboard.stats'));

        // Assert successful JSON response
        $response->assertStatus(200);
        $response->assertJsonStructure([
            'labels',
            'data',
        ]);

        // Assert that we have data
        $responseData = $response->json();
        $this->assertNotEmpty($responseData['labels'], 'Labels should not be empty');
        $this->assertNotEmpty($responseData['data'], 'Data should not be empty');
        $this->assertEquals(count($responseData['labels']), count($responseData['data']), 'Labels and data should have the same count');
    }

    /**
     * Test that provider pages load correctly.
     *
     * @return void
     */
    public function test_provider_pages_load()
    {
        // Seed demo data
        $this->seed(SocialDemoSeeder::class);

        // Get the demo creator user
        $user = User::where('email', 'creator1@demo.local')->first();

        // Ensure the user exists
        $this->assertNotNull($user, 'Demo creator user not found');

        // Login as the creator user
        $response = $this->actingAs($user);

        // Test a few provider pages
        $providers = ['youtube', 'instagram', 'twitter'];

        foreach ($providers as $provider) {
            // Visit the provider page
            $response = $this->get(route('creator.provider.show', $provider));

            // Assert successful response
            $response->assertStatus(200);

            // Assert that provider name is present
            $providerName = \App\Services\SocialProviders\ProviderFactory::getProviderName($provider);
            $response->assertSee($providerName . ' Posts');

            // Assert that posts table is present
            $response->assertSee('Posts from ' . $providerName);
        }
    }
}