<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use App\Models\User;
use App\Models\MessageThread;
use App\Models\Message;
use App\Models\MessageThreadParticipant;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class MessagingFeatureTest extends TestCase
{
    use RefreshDatabase;
    
    protected $brandUser;
    protected $creatorUser;
    protected $thread;
    
    protected function setUp(): void
    {
        parent::setUp();
        
        // Create users
        $this->brandUser = User::factory()->create();
        $this->brandUser->assignRole('brand_admin');
        
        $this->creatorUser = User::factory()->create();
        $this->creatorUser->assignRole('creator');
        
        // Create a thread
        $this->thread = MessageThread::create([
            'subject' => 'Test Thread',
        ]);
        
        // Add participants
        MessageThreadParticipant::create([
            'thread_id' => $this->thread->id,
            'user_id' => $this->brandUser->id,
            'joined_at' => now(),
        ]);
        
        MessageThreadParticipant::create([
            'thread_id' => $this->thread->id,
            'user_id' => $this->creatorUser->id,
            'joined_at' => now(),
        ]);
    }
    
    /** @test */
    public function users_can_view_their_message_threads()
    {
        // Create some messages in the thread
        Message::create([
            'thread_id' => $this->thread->id,
            'sender_id' => $this->brandUser->id,
            'recipient_id' => $this->creatorUser->id,
            'message' => 'Hello creator!',
        ]);
        
        // Brand user can view threads
        $response = $this->actingAs($this->brandUser)
            ->get(route('messages.index'));
            
        $response->assertStatus(200);
        $response->assertSee($this->thread->subject);
        $response->assertSee('Hello creator!');
        
        // Creator user can view threads
        $response = $this->actingAs($this->creatorUser)
            ->get(route('messages.index'));
            
        $response->assertStatus(200);
        $response->assertSee($this->thread->subject);
    }
    
    /** @test */
    public function users_can_view_a_specific_thread()
    {
        // Create a message
        $message = Message::create([
            'thread_id' => $this->thread->id,
            'sender_id' => $this->brandUser->id,
            'recipient_id' => $this->creatorUser->id,
            'message' => 'Hello creator!',
        ]);
        
        // Brand user can view thread
        $response = $this->actingAs($this->brandUser)
            ->get(route('messages.show', $this->thread));
            
        $response->assertStatus(200);
        $response->assertSee($message->message);
        
        // Creator user can view thread
        $response = $this->actingAs($this->creatorUser)
            ->get(route('messages.show', $this->thread));
            
        $response->assertStatus(200);
        $response->assertSee($message->message);
    }
    
    /** @test */
    public function unauthorized_users_cannot_view_threads()
    {
        // Create another user
        $otherUser = User::factory()->create();
        $otherUser->assignRole('creator');
        
        // Other user cannot view thread
        $response = $this->actingAs($otherUser)
            ->get(route('messages.show', $this->thread));
            
        $response->assertStatus(403);
    }
    
    /** @test */
    public function users_can_send_messages_with_text_only()
    {
        Mail::fake();
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('messages.send', $this->thread), [
                'body' => 'Test message body',
            ]);
            
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('messages', [
            'thread_id' => $this->thread->id,
            'sender_id' => $this->brandUser->id,
            'message' => 'Test message body',
        ]);
        
        // Verify notification was sent
        Mail::assertSent(\App\Mail\MessageReceived::class);
    }
    
    /** @test */
    public function users_can_send_messages_with_attachments()
    {
        Mail::fake();
        Storage::fake('public');
        
        $file = UploadedFile::fake()->image('test.jpg')->size(1000); // 1MB
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('messages.send', $this->thread), [
                'body' => 'Test message with attachment',
                'attachment' => $file,
            ]);
            
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('messages', [
            'thread_id' => $this->thread->id,
            'sender_id' => $this->brandUser->id,
            'message' => 'Test message with attachment',
        ]);
        
        // Verify file was stored
        Storage::disk('public')->assertExists('messages');
        
        // Verify notification was sent
        Mail::assertSent(\App\Mail\MessageReceived::class);
    }
    
    /** @test */
    public function attachment_validation_works_correctly()
    {
        Mail::fake();
        
        // Test file too large
        $largeFile = UploadedFile::fake()->create('test.pdf', 6000); // 6MB
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('messages.send', $this->thread), [
                'body' => 'Test message with large attachment',
                'attachment' => $largeFile,
            ]);
            
        $response->assertSessionHasErrors('attachment');
        
        // Test invalid file type
        $invalidFile = UploadedFile::fake()->create('test.exe', 1000);
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('messages.send', $this->thread), [
                'body' => 'Test message with invalid attachment',
                'attachment' => $invalidFile,
            ]);
            
        $response->assertSessionHasErrors('attachment');
    }
    
    /** @test */
    public function message_body_is_required()
    {
        $response = $this->actingAs($this->brandUser)
            ->post(route('messages.send', $this->thread), [
                'body' => '',
            ]);
            
        $response->assertSessionHasErrors('body');
    }
    
    /** @test */
    public function users_can_create_new_threads()
    {
        Mail::fake();
        
        $response = $this->actingAs($this->brandUser)
            ->post(route('threads.create'), [
                'recipient_id' => $this->creatorUser->id,
                'body' => 'Starting a new conversation',
            ]);
            
        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('message_threads', [
            'subject' => 'Proposal Discussion',
        ]);
        
        $this->assertDatabaseHas('messages', [
            'sender_id' => $this->brandUser->id,
            'recipient_id' => $this->creatorUser->id,
            'message' => 'Starting a new conversation',
        ]);
        
        // Verify notification was sent
        Mail::assertSent(\App\Mail\MessageReceived::class);
    }
}