<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Proposal;
use App\Models\Contract;
use App\Models\Payment;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class NotificationTest extends TestCase
{
    use RefreshDatabase;

    protected $brandAdmin;
    protected $creator;

    protected function setUp(): void
    {
        parent::setUp();

        // Create roles if they don't exist
        if (!Role::where('name', 'brand_admin')->exists()) {
            Role::create(['name' => 'brand_admin']);
        }
        
        if (!Role::where('name', 'creator')->exists()) {
            Role::create(['name' => 'creator']);
        }

        // Create a brand admin user
        $this->brandAdmin = User::factory()->create([
            'name' => 'Test Brand Admin',
            'email' => 'brand@test.com',
        ]);
        $this->brandAdmin->assignRole('brand_admin');

        // Create a creator user
        $this->creator = User::factory()->create([
            'name' => 'Test Creator',
            'email' => 'creator@test.com',
        ]);
        $this->creator->assignRole('creator');
    }

    public function test_proposal_accepted_notification()
    {
        // Create a proposal
        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $this->creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'sent',
        ]);

        // Accept the proposal
        $response = $this->actingAs($this->brandAdmin)
            ->put(route('brand.proposals.accept', $proposal));
        
        $response->assertStatus(302);
        
        // Check that the creator received a notification
        $this->assertCount(1, $this->creator->notifications);
        $this->assertEquals('Your proposal has been accepted: Test Proposal', $this->creator->notifications->first()->data['message']);
    }

    public function test_contract_sent_notification()
    {
        // Create a proposal
        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $this->creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'accepted',
        ]);

        // Create a contract
        $contract = Contract::factory()->create([
            'proposal_id' => $proposal->id,
            'brand_id' => $proposal->brand_id,
            'creator_id' => $proposal->creator_id,
            'title' => 'Contract for Test Proposal',
            'status' => 'draft',
        ]);

        // Update proposal with contract
        $proposal->update(['contract_id' => $contract->id]);

        // Send the contract
        $response = $this->actingAs($this->brandAdmin)
            ->post(route('brand.proposals.send-contract', $proposal));
        
        $response->assertStatus(302);
        
        // Check that the creator received a notification
        $this->assertCount(1, $this->creator->notifications);
        $this->assertEquals('A contract has been sent for your proposal: Test Proposal', $this->creator->notifications->first()->data['message']);
    }

    public function test_contract_signed_notification()
    {
        // Create a proposal
        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $this->creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'contract_sent',
        ]);

        // Create a contract
        $contract = Contract::factory()->create([
            'proposal_id' => $proposal->id,
            'brand_id' => $proposal->brand_id,
            'creator_id' => $proposal->creator_id,
            'title' => 'Contract for Test Proposal',
            'status' => 'sent',
        ]);

        // Update proposal with contract
        $proposal->update(['contract_id' => $contract->id]);

        // Mark contract as signed
        $response = $this->actingAs($this->brandAdmin)
            ->post(route('brand.proposals.contract-signed', $proposal));
        
        $response->assertStatus(302);
        
        // Check that the brand admin received a notification
        $this->assertCount(1, $this->brandAdmin->notifications);
        $this->assertEquals('The contract for your proposal has been signed: Test Proposal', $this->brandAdmin->notifications->first()->data['message']);
    }

    public function test_payment_processed_notification()
    {
        // Create a proposal
        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $this->creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'contract_signed',
        ]);

        // Create a contract
        $contract = Contract::factory()->create([
            'proposal_id' => $proposal->id,
            'brand_id' => $proposal->brand_id,
            'creator_id' => $proposal->creator_id,
            'title' => 'Contract for Test Proposal',
            'status' => 'signed',
            'value' => 1000,
        ]);

        // Update proposal with contract
        $proposal->update(['contract_id' => $contract->id]);

        // Process payment
        $response = $this->actingAs($this->brandAdmin)
            ->post(route('brand.proposals.process-payment', $proposal));
        
        $response->assertStatus(302);
        
        // Check that the creator received a notification
        $this->assertCount(1, $this->creator->notifications);
        $this->assertEquals('The payment for your proposal has been processed: Test Proposal', $this->creator->notifications->first()->data['message']);
    }

    public function test_work_delivered_notification()
    {
        // Create a proposal
        $proposal = Proposal::factory()->create([
            'brand_id' => 1,
            'creator_id' => $this->creator->id,
            'title' => 'Test Proposal',
            'message' => 'This is a test proposal message',
            'type' => 'Collaboration',
            'status' => 'payment_made',
        ]);

        // Mark work as delivered
        $response = $this->actingAs($this->brandAdmin)
            ->post(route('brand.proposals.mark-delivered', $proposal));
        
        $response->assertStatus(302);
        
        // Check that the brand admin received a notification
        $this->assertCount(1, $this->brandAdmin->notifications);
        $this->assertEquals('The work for your proposal has been delivered: Test Proposal', $this->brandAdmin->notifications->first()->data['message']);
    }
}