<?php

namespace Tests\Feature;

use App\Models\Brand;
use App\Models\Proposal;
use App\Models\ProposalComment;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class ProposalChatTest extends TestCase
{
    use RefreshDatabase;

    protected $brandUser;
    protected $creatorUser;
    protected $brand;
    protected $proposal;

    protected function setUp(): void
    {
        parent::setUp();

        // Create a brand
        $this->brand = Brand::factory()->create();

        // Create a brand admin user
        $this->brandUser = User::factory()->create([
            'brand_id' => $this->brand->id,
        ]);
        $this->brandUser->assignRole('brand_admin');

        // Create a creator user
        $this->creatorUser = User::factory()->create();
        $this->creatorUser->assignRole('creator');

        // Create a proposal
        $this->proposal = Proposal::factory()->create([
            'brand_id' => $this->brand->id,
            'creator_id' => $this->creatorUser->id,
        ]);
    }

    /** @test */
    public function brand_admin_can_add_comment_to_proposal()
    {
        $response = $this->actingAs($this->brandUser)
            ->post(route('brand.proposals.comments.store', $this->proposal), [
                'message' => 'This is a test comment from brand admin',
            ]);

        $response->assertStatus(302);
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('proposal_comments', [
            'proposal_id' => $this->proposal->id,
            'user_id' => $this->brandUser->id,
            'message' => 'This is a test comment from brand admin',
        ]);
    }

    /** @test */
    public function creator_can_add_comment_to_proposal()
    {
        $response = $this->actingAs($this->creatorUser)
            ->post(route('creator.proposals.comments.store', $this->proposal), [
                'message' => 'This is a test comment from creator',
            ]);

        $response->assertStatus(302);
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('proposal_comments', [
            'proposal_id' => $this->proposal->id,
            'user_id' => $this->creatorUser->id,
            'message' => 'This is a test comment from creator',
        ]);
    }

    /** @test */
    public function brand_admin_can_view_proposal_comments()
    {
        // Create a comment
        $comment = ProposalComment::factory()->create([
            'proposal_id' => $this->proposal->id,
            'user_id' => $this->brandUser->id,
            'message' => 'Test comment',
        ]);

        $response = $this->actingAs($this->brandUser)
            ->get(route('brand.proposals.comments.index', $this->proposal));

        $response->assertStatus(200);
        $response->assertJsonFragment([
            'message' => 'Test comment',
        ]);
    }

    /** @test */
    public function creator_can_view_proposal_comments()
    {
        // Create a comment
        $comment = ProposalComment::factory()->create([
            'proposal_id' => $this->proposal->id,
            'user_id' => $this->creatorUser->id,
            'message' => 'Test comment',
        ]);

        $response = $this->actingAs($this->creatorUser)
            ->get(route('creator.proposals.comments.index', $this->proposal));

        $response->assertStatus(200);
        $response->assertJsonFragment([
            'message' => 'Test comment',
        ]);
    }

    /** @test */
    public function unauthorized_user_cannot_add_comment_to_proposal()
    {
        // Create another user
        $otherUser = User::factory()->create();
        $otherUser->assignRole('creator');

        $response = $this->actingAs($otherUser)
            ->post(route('brand.proposals.comments.store', $this->proposal), [
                'message' => 'This should fail',
            ]);

        $response->assertStatus(403);
    }

    /** @test */
    public function unauthorized_user_cannot_view_proposal_comments()
    {
        // Create another user
        $otherUser = User::factory()->create();
        $otherUser->assignRole('creator');

        $response = $this->actingAs($otherUser)
            ->get(route('brand.proposals.comments.index', $this->proposal));

        $response->assertStatus(403);
    }
}