"""
Health check and status endpoints
"""
from fastapi import APIRouter, Depends
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import text
import redis.asyncio as redis
from datetime import datetime

from app.database import get_db
from app.schemas import HealthResponse
from config import settings
from app.utils.observability import logger

router = APIRouter()


async def check_database() -> str:
    """Check database connectivity"""
    try:
        async with get_db() as db:
            await db.execute(text("SELECT 1"))
        return "healthy"
    except Exception as e:
        logger.error("Database health check failed", error=str(e))
        return "unhealthy"


async def check_redis() -> str:
    """Check Redis connectivity"""
    try:
        r = redis.from_url(settings.redis_url)
        await r.ping()
        await r.close()
        return "healthy"
    except Exception as e:
        logger.error("Redis health check failed", error=str(e))
        return "unhealthy"


async def check_celery() -> str:
    """Check Celery workers"""
    try:
        from app.celery_app import celery_app
        inspect = celery_app.control.inspect()
        active_workers = inspect.active()
        if active_workers:
            return "healthy"
        return "no_workers"
    except Exception as e:
        logger.error("Celery health check failed", error=str(e))
        return "unhealthy"


async def check_vector_store() -> str:
    """Check vector store connectivity"""
    try:
        if settings.vector_store_type == "faiss":
            # FAISS is file-based, check if index exists
            import os
            if os.path.exists(settings.faiss_index_path):
                return "healthy"
            return "not_initialized"
        elif settings.vector_store_type == "weaviate":
            # Check Weaviate connectivity
            import weaviate
            client = weaviate.Client(settings.weaviate_url)
            client.schema.get()
            return "healthy"
        return "unknown"
    except Exception as e:
        logger.error("Vector store health check failed", error=str(e))
        return "unhealthy"


@router.get("", response_model=HealthResponse)
async def health_check():
    """
    Comprehensive health check
    Returns status of all services
    """
    services = {
        "database": await check_database(),
        "redis": await check_redis(),
        "celery": await check_celery(),
        "vector_store": await check_vector_store(),
    }
    
    # Overall status is healthy if all critical services are healthy
    critical_services = ["database", "redis"]
    overall_status = "healthy" if all(
        services[s] == "healthy" for s in critical_services
    ) else "degraded"
    
    return HealthResponse(
        status=overall_status,
        version="1.0.0",
        timestamp=datetime.utcnow(),
        services=services
    )


@router.get("/liveness")
async def liveness():
    """
    Kubernetes liveness probe
    Returns 200 if service is running
    """
    return {"status": "alive"}


@router.get("/readiness")
async def readiness():
    """
    Kubernetes readiness probe
    Returns 200 if service is ready to accept traffic
    """
    db_status = await check_database()
    redis_status = await check_redis()
    
    if db_status == "healthy" and redis_status == "healthy":
        return {"status": "ready"}
    
    from fastapi import status
    from fastapi.responses import JSONResponse
    return JSONResponse(
        status_code=status.HTTP_503_SERVICE_UNAVAILABLE,
        content={"status": "not_ready", "database": db_status, "redis": redis_status}
    )