"""
Pydantic schemas for request/response validation
"""
from typing import Optional, Dict, Any, List, Literal
from pydantic import BaseModel, Field, validator
from datetime import datetime
from enum import Enum


# Enums
class JobType(str, Enum):
    """AI Job types"""
    PROPOSAL = "proposal"
    CONTENT_IDEAS = "content_ideas"
    AUTO_REPLY = "auto_reply"
    ADS_GENERATE = "ads_generate"
    FORECAST = "forecast"
    MATCHMAKING = "matchmaking"
    TRANSLATE = "translate"
    CONTRACT_DRAFT = "contract_draft"
    SUPPORT_RAG = "support_rag"
    CUSTOM = "custom"


class JobStatus(str, Enum):
    """Job status values"""
    PENDING = "pending"
    PROCESSING = "processing"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"


class SocialPlatform(str, Enum):
    """Social media platforms"""
    TWITTER = "twitter"
    FACEBOOK = "facebook"
    INSTAGRAM = "instagram"
    YOUTUBE = "youtube"
    LINKEDIN = "linkedin"


class PublishStatus(str, Enum):
    """Publish job status"""
    PENDING = "pending"
    PUBLISHING = "publishing"
    PUBLISHED = "published"
    FAILED = "failed"


# Request Schemas
class AIJobRequest(BaseModel):
    """Request to create an AI job"""
    job_type: JobType = Field(..., description="Type of AI job to execute")
    tenant_id: str = Field(..., description="Brand or creator ID")
    owner_id: str = Field(..., description="User ID in Laravel")
    payload: Dict[str, Any] = Field(..., description="Job-specific parameters")
    callback_url: Optional[str] = Field(None, description="URL to POST results")
    idempotency_key: Optional[str] = Field(None, description="Unique key for idempotency")
    priority: Optional[int] = Field(5, ge=1, le=10, description="Job priority (1-10)")
    
    class Config:
        json_schema_extra = {
            "example": {
                "job_type": "content_ideas",
                "tenant_id": "brand-123",
                "owner_id": "user-456",
                "payload": {
                    "topic": "AI in marketing",
                    "count": 5,
                    "tone": "professional"
                },
                "callback_url": "https://laravel.example/api/ai-callback",
                "idempotency_key": "unique-key-123"
            }
        }


class AIJobResponse(BaseModel):
    """Response after creating an AI job"""
    ai_job_id: str = Field(..., description="Unique job identifier")
    status: JobStatus = Field(..., description="Current job status")
    estimated_time_seconds: Optional[int] = Field(None, description="Estimated completion time")
    message: str = Field(..., description="Human-readable message")
    
    class Config:
        json_schema_extra = {
            "example": {
                "ai_job_id": "job-789",
                "status": "pending",
                "estimated_time_seconds": 30,
                "message": "Job queued successfully"
            }
        }


class JobStatusResponse(BaseModel):
    """Response for job status query"""
    ai_job_id: str
    job_type: str
    status: JobStatus
    result: Optional[Dict[str, Any]] = None
    error_message: Optional[str] = None
    provider: Optional[str] = None
    model_used: Optional[str] = None
    tokens_input: int = 0
    tokens_output: int = 0
    cost_usd: float = 0.0
    created_at: datetime
    started_at: Optional[datetime] = None
    completed_at: Optional[datetime] = None
    metadata: Optional[Dict[str, Any]] = None


class PublishRequest(BaseModel):
    """Request to publish content to social media"""
    platform: SocialPlatform = Field(..., description="Target social platform")
    tenant_id: str = Field(..., description="Brand or creator ID")
    owner_id: str = Field(..., description="User ID")
    content: Dict[str, Any] = Field(..., description="Content to publish")
    callback_url: Optional[str] = Field(None, description="Callback URL")
    idempotency_key: Optional[str] = Field(None, description="Idempotency key")
    schedule_at: Optional[datetime] = Field(None, description="Schedule for future")
    
    class Config:
        json_schema_extra = {
            "example": {
                "platform": "twitter",
                "tenant_id": "brand-123",
                "owner_id": "user-456",
                "content": {
                    "text": "Check out our new product!",
                    "media_urls": ["https://example.com/image.jpg"]
                },
                "callback_url": "https://laravel.example/api/publish-callback"
            }
        }


class PublishResponse(BaseModel):
    """Response after publish request"""
    publish_job_id: str
    status: PublishStatus
    platform: str
    message: str
    platform_post_id: Optional[str] = None


class ReelProcessRequest(BaseModel):
    """Request to process a reel/video"""
    tenant_id: str
    owner_id: str
    video_url: str = Field(..., description="URL of video to process")
    processing_options: Dict[str, Any] = Field(
        default_factory=dict,
        description="Processing options (transcription, captions, etc.)"
    )
    callback_url: Optional[str] = None
    idempotency_key: Optional[str] = None


class ReelProcessResponse(BaseModel):
    """Response after reel process request"""
    reel_job_id: str
    status: str
    message: str
    estimated_time_seconds: Optional[int] = None


class RAGIngestRequest(BaseModel):
    """Request to ingest documents for RAG"""
    tenant_id: str
    documents: List[Dict[str, Any]] = Field(
        ...,
        description="List of documents with id, title, content, metadata"
    )
    document_type: str = Field(default="knowledge_base")
    
    class Config:
        json_schema_extra = {
            "example": {
                "tenant_id": "brand-123",
                "documents": [
                    {
                        "id": "doc-1",
                        "title": "Product FAQ",
                        "content": "Q: What is the product? A: ...",
                        "metadata": {"category": "faq"}
                    }
                ],
                "document_type": "support_doc"
            }
        }


class RAGIngestResponse(BaseModel):
    """Response after RAG ingest"""
    ingested_count: int
    failed_count: int
    document_ids: List[str]
    message: str


class RAGQueryRequest(BaseModel):
    """Request to query RAG system"""
    tenant_id: str
    query: str = Field(..., description="User query")
    document_type: Optional[str] = Field(None, description="Filter by document type")
    top_k: int = Field(5, ge=1, le=20, description="Number of results")
    include_sources: bool = Field(True, description="Include source documents")


class RAGQueryResponse(BaseModel):
    """Response from RAG query"""
    answer: str
    sources: Optional[List[Dict[str, Any]]] = None
    confidence: Optional[float] = None
    tokens_used: int = 0
    cost_usd: float = 0.0


class CallbackPayload(BaseModel):
    """Payload sent to Laravel callback"""
    ai_job_id: str
    status: JobStatus
    result: Optional[Dict[str, Any]] = None
    error_message: Optional[str] = None
    provider: Optional[str] = None
    model_used: Optional[str] = None
    tokens_input: int = 0
    tokens_output: int = 0
    cost_usd: float = 0.0
    completed_at: datetime
    metadata: Optional[Dict[str, Any]] = None


class HealthResponse(BaseModel):
    """Health check response"""
    status: str
    version: str
    timestamp: datetime
    services: Dict[str, str]
    
    class Config:
        json_schema_extra = {
            "example": {
                "status": "healthy",
                "version": "1.0.0",
                "timestamp": "2024-01-01T00:00:00Z",
                "services": {
                    "database": "healthy",
                    "redis": "healthy",
                    "celery": "healthy",
                    "vector_store": "healthy"
                }
            }
        }


class MetricsResponse(BaseModel):
    """Metrics response"""
    total_jobs: int
    jobs_by_status: Dict[str, int]
    jobs_by_type: Dict[str, int]
    total_cost_usd: float
    avg_processing_time_seconds: float
    active_workers: int


class ErrorResponse(BaseModel):
    """Standard error response"""
    error: str
    detail: Optional[str] = None
    error_code: Optional[str] = None
    timestamp: datetime = Field(default_factory=datetime.utcnow)
    
    class Config:
        json_schema_extra = {
            "example": {
                "error": "Invalid request",
                "detail": "Missing required field: tenant_id",
                "error_code": "VALIDATION_ERROR",
                "timestamp": "2024-01-01T00:00:00Z"
            }
        }