"""
Publisher processor - Handles social media publishing jobs
"""
from typing import Dict, Any
from datetime import datetime
from sqlalchemy import select

from app.database import get_db_session
from app.models import PublishJob
from app.services.social_publishers import get_publisher
from app.utils.observability import logger, metrics_collector
from app.celery_app import send_callback
from config import settings


async def process_publish(job_id: str) -> Dict[str, Any]:
    """
    Process a social media publish job
    
    Handles publishing to various social platforms
    """
    async with get_db_session() as db:
        # Get job
        stmt = select(PublishJob).where(PublishJob.id == job_id)
        result = await db.execute(stmt)
        job = result.scalar_one_or_none()
        
        if not job:
            raise ValueError(f"Publish job {job_id} not found")
        
        # Update status to publishing
        job.status = "publishing"
        await db.commit()
        
        logger.info(
            "Processing publish job",
            job_id=job_id,
            platform=job.platform
        )
        
        start_time = datetime.utcnow()
        
        try:
            # Get publisher for platform
            publisher = get_publisher(job.platform)
            
            # Publish content
            result = await publisher.publish(job.content)
            
            # Update job with result
            job.status = "published"
            job.platform_post_id = result.get("platform_post_id")
            job.platform_response = result
            job.published_at = datetime.utcnow()
            
            await db.commit()
            
            # Track metrics
            duration = (job.published_at - start_time).total_seconds()
            metrics_collector.track_publish_request(job.platform, "published", duration)
            
            logger.info(
                "Publish job completed",
                job_id=job_id,
                platform=job.platform,
                post_id=result.get("platform_post_id")
            )
            
            # Send callback if configured
            if job.callback_url:
                await send_publish_callback(job)
            
            return {
                "status": "published",
                "job_id": job_id,
                "platform_post_id": result.get("platform_post_id")
            }
            
        except Exception as e:
            # Mark job as failed
            job.status = "failed"
            job.error_message = str(e)
            await db.commit()
            
            # Track metrics
            duration = (datetime.utcnow() - start_time).total_seconds()
            metrics_collector.track_publish_request(job.platform, "failed", duration)
            
            logger.error(
                "Publish job failed",
                job_id=job_id,
                platform=job.platform,
                error=str(e)
            )
            
            # Send failure callback
            if job.callback_url:
                await send_publish_callback(job)
            
            raise


async def send_publish_callback(job: PublishJob):
    """Send publish callback to Laravel"""
    payload = {
        "publish_job_id": job.id,
        "platform": job.platform,
        "status": job.status,
        "platform_post_id": job.platform_post_id,
        "platform_response": job.platform_response,
        "error_message": job.error_message,
        "published_at": job.published_at.isoformat() if job.published_at else None,
        "metadata": job.metadata
    }
    
    # Enqueue callback task
    send_callback.apply_async(
        args=[job.callback_url, payload],
        retry=True
    )
    
    logger.info(
        "Publish callback enqueued",
        job_id=job.id,
        callback_url=job.callback_url
    )