"""
Social media publisher adapters
Handles publishing to Twitter, Facebook, Instagram, YouTube, LinkedIn
"""
from typing import Dict, Any, Optional, List
from abc import ABC, abstractmethod
import time

from config import settings
from app.utils.observability import logger


class SocialPublisher(ABC):
    """Base class for social media publishers"""
    
    def __init__(self, platform_name: str):
        self.platform_name = platform_name
    
    @abstractmethod
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Publish content to platform"""
        pass
    
    @abstractmethod
    async def delete_post(self, post_id: str) -> bool:
        """Delete a post"""
        pass
    
    @abstractmethod
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Get post status"""
        pass


class TwitterPublisher(SocialPublisher):
    """Twitter/X publisher"""
    
    def __init__(self):
        super().__init__("twitter")
        self.api_key = settings.twitter_api_key
        self.api_secret = settings.twitter_api_secret
        self.access_token = settings.twitter_access_token
        self.access_secret = settings.twitter_access_secret
        
        if not all([self.api_key, self.api_secret, self.access_token, self.access_secret]):
            raise ValueError("Twitter credentials not fully configured")
    
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Publish tweet"""
        import tweepy
        
        try:
            # Initialize Twitter client
            client = tweepy.Client(
                consumer_key=self.api_key,
                consumer_secret=self.api_secret,
                access_token=self.access_token,
                access_token_secret=self.access_secret
            )
            
            text = content.get("text", "")
            media_urls = content.get("media_urls", [])
            
            # Upload media if present
            media_ids = []
            if media_urls:
                auth = tweepy.OAuth1UserHandler(
                    self.api_key, self.api_secret,
                    self.access_token, self.access_secret
                )
                api = tweepy.API(auth)
                
                for media_url in media_urls[:4]:  # Twitter allows max 4 images
                    try:
                        import httpx
                        async with httpx.AsyncClient() as http_client:
                            response = await http_client.get(media_url)
                            response.raise_for_status()
                            media_data = response.content
                        
                        media = api.media_upload(filename="temp", file=media_data)
                        media_ids.append(media.media_id)
                    except Exception as e:
                        logger.error("Failed to upload media to Twitter", error=str(e))
            
            # Create tweet
            response = client.create_tweet(
                text=text,
                media_ids=media_ids if media_ids else None
            )
            
            tweet_id = response.data["id"]
            
            logger.info("Tweet published", tweet_id=tweet_id)
            
            return {
                "platform_post_id": tweet_id,
                "url": f"https://twitter.com/i/web/status/{tweet_id}",
                "status": "published"
            }
            
        except Exception as e:
            logger.error("Failed to publish tweet", error=str(e))
            raise
    
    async def delete_post(self, post_id: str) -> bool:
        """Delete tweet"""
        import tweepy
        
        try:
            client = tweepy.Client(
                consumer_key=self.api_key,
                consumer_secret=self.api_secret,
                access_token=self.access_token,
                access_token_secret=self.access_secret
            )
            
            client.delete_tweet(post_id)
            logger.info("Tweet deleted", tweet_id=post_id)
            return True
            
        except Exception as e:
            logger.error("Failed to delete tweet", tweet_id=post_id, error=str(e))
            return False
    
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Get tweet status"""
        import tweepy
        
        try:
            client = tweepy.Client(
                consumer_key=self.api_key,
                consumer_secret=self.api_secret,
                access_token=self.access_token,
                access_token_secret=self.access_secret
            )
            
            tweet = client.get_tweet(post_id)
            
            return {
                "exists": True,
                "text": tweet.data.text,
                "created_at": tweet.data.created_at
            }
            
        except Exception as e:
            logger.error("Failed to get tweet status", tweet_id=post_id, error=str(e))
            return {"exists": False}


class FacebookPublisher(SocialPublisher):
    """Facebook publisher"""
    
    def __init__(self):
        super().__init__("facebook")
        self.app_id = settings.facebook_app_id
        self.app_secret = settings.facebook_app_secret
        self.access_token = settings.facebook_access_token
        
        if not all([self.app_id, self.app_secret, self.access_token]):
            raise ValueError("Facebook credentials not fully configured")
    
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Publish to Facebook page"""
        import facebook
        
        try:
            graph = facebook.GraphAPI(access_token=self.access_token)
            
            message = content.get("text", "")
            page_id = content.get("page_id")  # Required
            media_urls = content.get("media_urls", [])
            
            if not page_id:
                raise ValueError("page_id required for Facebook publishing")
            
            # Publish post
            if media_urls:
                # Post with photo
                response = graph.put_photo(
                    image=media_urls[0],
                    message=message,
                    album_path=f"{page_id}/photos"
                )
            else:
                # Text post
                response = graph.put_object(
                    parent_object=page_id,
                    connection_name="feed",
                    message=message
                )
            
            post_id = response.get("id") or response.get("post_id")
            
            logger.info("Facebook post published", post_id=post_id)
            
            return {
                "platform_post_id": post_id,
                "url": f"https://facebook.com/{post_id}",
                "status": "published"
            }
            
        except Exception as e:
            logger.error("Failed to publish to Facebook", error=str(e))
            raise
    
    async def delete_post(self, post_id: str) -> bool:
        """Delete Facebook post"""
        import facebook
        
        try:
            graph = facebook.GraphAPI(access_token=self.access_token)
            graph.delete_object(post_id)
            logger.info("Facebook post deleted", post_id=post_id)
            return True
            
        except Exception as e:
            logger.error("Failed to delete Facebook post", post_id=post_id, error=str(e))
            return False
    
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Get Facebook post status"""
        import facebook
        
        try:
            graph = facebook.GraphAPI(access_token=self.access_token)
            post = graph.get_object(post_id)
            
            return {
                "exists": True,
                "message": post.get("message"),
                "created_time": post.get("created_time")
            }
            
        except Exception as e:
            logger.error("Failed to get Facebook post status", post_id=post_id, error=str(e))
            return {"exists": False}


class InstagramPublisher(SocialPublisher):
    """Instagram publisher (via Facebook Graph API)"""
    
    def __init__(self):
        super().__init__("instagram")
        self.access_token = settings.facebook_access_token
        
        if not self.access_token:
            raise ValueError("Instagram/Facebook access token not configured")
    
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Publish to Instagram"""
        import httpx
        
        try:
            instagram_account_id = content.get("instagram_account_id")
            if not instagram_account_id:
                raise ValueError("instagram_account_id required")
            
            caption = content.get("text", "")
            image_url = content.get("media_urls", [])[0] if content.get("media_urls") else None
            
            if not image_url:
                raise ValueError("Instagram requires at least one image")
            
            # Create media container
            async with httpx.AsyncClient() as client:
                container_response = await client.post(
                    f"https://graph.facebook.com/v18.0/{instagram_account_id}/media",
                    params={
                        "image_url": image_url,
                        "caption": caption,
                        "access_token": self.access_token
                    }
                )
                container_response.raise_for_status()
                container_id = container_response.json()["id"]
                
                # Publish media
                publish_response = await client.post(
                    f"https://graph.facebook.com/v18.0/{instagram_account_id}/media_publish",
                    params={
                        "creation_id": container_id,
                        "access_token": self.access_token
                    }
                )
                publish_response.raise_for_status()
                post_id = publish_response.json()["id"]
            
            logger.info("Instagram post published", post_id=post_id)
            
            return {
                "platform_post_id": post_id,
                "url": f"https://instagram.com/p/{post_id}",
                "status": "published"
            }
            
        except Exception as e:
            logger.error("Failed to publish to Instagram", error=str(e))
            raise
    
    async def delete_post(self, post_id: str) -> bool:
        """Delete Instagram post"""
        import httpx
        
        try:
            async with httpx.AsyncClient() as client:
                response = await client.delete(
                    f"https://graph.facebook.com/v18.0/{post_id}",
                    params={"access_token": self.access_token}
                )
                response.raise_for_status()
            
            logger.info("Instagram post deleted", post_id=post_id)
            return True
            
        except Exception as e:
            logger.error("Failed to delete Instagram post", post_id=post_id, error=str(e))
            return False
    
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Get Instagram post status"""
        import httpx
        
        try:
            async with httpx.AsyncClient() as client:
                response = await client.get(
                    f"https://graph.facebook.com/v18.0/{post_id}",
                    params={
                        "fields": "caption,timestamp",
                        "access_token": self.access_token
                    }
                )
                response.raise_for_status()
                data = response.json()
            
            return {
                "exists": True,
                "caption": data.get("caption"),
                "timestamp": data.get("timestamp")
            }
            
        except Exception as e:
            logger.error("Failed to get Instagram post status", post_id=post_id, error=str(e))
            return {"exists": False}


class LinkedInPublisher(SocialPublisher):
    """LinkedIn publisher"""
    
    def __init__(self):
        super().__init__("linkedin")
        self.access_token = settings.linkedin_access_token
        
        if not self.access_token:
            raise ValueError("LinkedIn access token not configured")
    
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Publish to LinkedIn"""
        import httpx
        
        try:
            person_urn = content.get("person_urn")  # e.g., "urn:li:person:ABC123"
            if not person_urn:
                raise ValueError("person_urn required for LinkedIn")
            
            text = content.get("text", "")
            
            post_data = {
                "author": person_urn,
                "lifecycleState": "PUBLISHED",
                "specificContent": {
                    "com.linkedin.ugc.ShareContent": {
                        "shareCommentary": {
                            "text": text
                        },
                        "shareMediaCategory": "NONE"
                    }
                },
                "visibility": {
                    "com.linkedin.ugc.MemberNetworkVisibility": "PUBLIC"
                }
            }
            
            async with httpx.AsyncClient() as client:
                response = await client.post(
                    "https://api.linkedin.com/v2/ugcPosts",
                    json=post_data,
                    headers={
                        "Authorization": f"Bearer {self.access_token}",
                        "Content-Type": "application/json",
                        "X-Restli-Protocol-Version": "2.0.0"
                    }
                )
                response.raise_for_status()
                post_id = response.headers.get("X-RestLi-Id")
            
            logger.info("LinkedIn post published", post_id=post_id)
            
            return {
                "platform_post_id": post_id,
                "url": f"https://linkedin.com/feed/update/{post_id}",
                "status": "published"
            }
            
        except Exception as e:
            logger.error("Failed to publish to LinkedIn", error=str(e))
            raise
    
    async def delete_post(self, post_id: str) -> bool:
        """Delete LinkedIn post"""
        import httpx
        
        try:
            async with httpx.AsyncClient() as client:
                response = await client.delete(
                    f"https://api.linkedin.com/v2/ugcPosts/{post_id}",
                    headers={"Authorization": f"Bearer {self.access_token}"}
                )
                response.raise_for_status()
            
            logger.info("LinkedIn post deleted", post_id=post_id)
            return True
            
        except Exception as e:
            logger.error("Failed to delete LinkedIn post", post_id=post_id, error=str(e))
            return False
    
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Get LinkedIn post status"""
        # LinkedIn API doesn't easily support getting individual posts
        return {"exists": True, "note": "LinkedIn status check not fully implemented"}


class MockPublisher(SocialPublisher):
    """Mock publisher for testing"""
    
    def __init__(self, platform_name: str = "mock"):
        super().__init__(platform_name)
    
    async def publish(self, content: Dict[str, Any]) -> Dict[str, Any]:
        """Mock publish"""
        import uuid
        
        post_id = str(uuid.uuid4())
        logger.info("Mock publish", platform=self.platform_name, post_id=post_id)
        
        return {
            "platform_post_id": post_id,
            "url": f"https://mock.example.com/{post_id}",
            "status": "published"
        }
    
    async def delete_post(self, post_id: str) -> bool:
        """Mock delete"""
        logger.info("Mock delete", platform=self.platform_name, post_id=post_id)
        return True
    
    async def get_post_status(self, post_id: str) -> Dict[str, Any]:
        """Mock status"""
        return {"exists": True, "status": "published"}


def get_publisher(platform: str) -> SocialPublisher:
    """
    Get social media publisher instance
    
    Args:
        platform: Platform name (twitter, facebook, instagram, linkedin)
    
    Returns:
        SocialPublisher instance
    """
    if settings.mock_providers or settings.test_mode:
        return MockPublisher(platform)
    
    publishers = {
        "twitter": TwitterPublisher,
        "facebook": FacebookPublisher,
        "instagram": InstagramPublisher,
        "linkedin": LinkedInPublisher,
    }
    
    publisher_class = publishers.get(platform)
    if not publisher_class:
        raise ValueError(f"Unknown platform: {platform}")
    
    try:
        return publisher_class()
    except ValueError as e:
        logger.warning(
            f"Failed to initialize {platform} publisher: {e}. Using mock publisher."
        )
        return MockPublisher(platform)