<?php
/**
 * Laravel-Python Integration Test Script
 * Run this script to verify the connection between Laravel and Python microservice
 * 
 * Usage: php test_integration.php
 */

require __DIR__ . '/Laravel/vendor/autoload.php';

use Illuminate\Support\Facades\Http;

// Colors for terminal output
class Colors {
    public static $GREEN = "\033[0;32m";
    public static $RED = "\033[0;31m";
    public static $YELLOW = "\033[1;33m";
    public static $BLUE = "\033[0;34m";
    public static $NC = "\033[0m"; // No Color
}

function printHeader($text) {
    echo "\n" . Colors::$BLUE . "========================================" . Colors::$NC . "\n";
    echo Colors::$BLUE . $text . Colors::$NC . "\n";
    echo Colors::$BLUE . "========================================" . Colors::$NC . "\n\n";
}

function printSuccess($text) {
    echo Colors::$GREEN . "✓ " . $text . Colors::$NC . "\n";
}

function printError($text) {
    echo Colors::$RED . "✗ " . $text . Colors::$NC . "\n";
}

function printInfo($text) {
    echo Colors::$YELLOW . "ℹ " . $text . Colors::$NC . "\n";
}

// Load environment variables
$envFile = __DIR__ . '/Laravel/.env';
if (!file_exists($envFile)) {
    printError("Laravel .env file not found!");
    printInfo("Please copy .env.example to .env first:");
    printInfo("cd Laravel && cp .env.example .env");
    exit(1);
}

// Parse .env file
$env = parse_ini_file($envFile);
$pythonUrl = $env['PYTHON_API_URL'] ?? 'http://localhost:8000';
$pythonApiKey = $env['PYTHON_API_KEY'] ?? '';

printHeader("Laravel-Python Integration Test");

// Test 1: Check Python .env file
printInfo("Test 1: Checking Python .env file...");
$pythonEnvFile = __DIR__ . '/pytho/.env';
if (file_exists($pythonEnvFile)) {
    printSuccess("Python .env file exists");
    $pythonEnv = parse_ini_file($pythonEnvFile);
    
    if (isset($pythonEnv['LARAVEL_API_KEY']) && $pythonEnv['LARAVEL_API_KEY'] === $pythonApiKey) {
        printSuccess("API keys match between Laravel and Python");
    } else {
        printError("API keys don't match!");
        printInfo("Laravel PYTHON_API_KEY: " . substr($pythonApiKey, 0, 20) . "...");
        printInfo("Python LARAVEL_API_KEY: " . substr($pythonEnv['LARAVEL_API_KEY'] ?? '', 0, 20) . "...");
    }
} else {
    printError("Python .env file not found!");
    printInfo("Please copy pytho/.env.example to pytho/.env");
}

// Test 2: Check if Python service is running
printInfo("\nTest 2: Checking if Python service is running...");
try {
    $response = @file_get_contents($pythonUrl . '/health/readiness', false, stream_context_create([
        'http' => [
            'method' => 'GET',
            'header' => "X-API-Key: $pythonApiKey\r\n",
            'timeout' => 5,
        ],
    ]));
    
    if ($response !== false) {
        $data = json_decode($response, true);
        if (isset($data['status']) && $data['status'] === 'healthy') {
            printSuccess("Python service is running and healthy");
            printSuccess("Database: " . ($data['database'] ?? 'unknown'));
            printSuccess("Redis: " . ($data['redis'] ?? 'unknown'));
        } else {
            printError("Python service responded but not healthy");
        }
    } else {
        printError("Cannot connect to Python service at: $pythonUrl");
        printInfo("Make sure Python service is running:");
        printInfo("cd pytho && docker-compose up -d");
    }
} catch (Exception $e) {
    printError("Error connecting to Python service: " . $e->getMessage());
}

// Test 3: Test API authentication
printInfo("\nTest 3: Testing API authentication...");
try {
    $ch = curl_init($pythonUrl . '/');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "X-API-Key: $pythonApiKey",
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        printSuccess("API authentication successful");
        $data = json_decode($response, true);
        printInfo("Service: " . ($data['service'] ?? 'unknown'));
        printInfo("Version: " . ($data['version'] ?? 'unknown'));
    } else {
        printError("API authentication failed (HTTP $httpCode)");
    }
} catch (Exception $e) {
    printError("Error testing authentication: " . $e->getMessage());
}

// Test 4: Check Laravel configuration
printInfo("\nTest 4: Checking Laravel configuration...");
$configFile = __DIR__ . '/Laravel/config/python.php';
if (file_exists($configFile)) {
    printSuccess("Laravel python.php config file exists");
} else {
    printError("Laravel python.php config file not found!");
}

$serviceFile = __DIR__ . '/Laravel/app/Services/PythonApiService.php';
if (file_exists($serviceFile)) {
    printSuccess("PythonApiService class exists");
} else {
    printError("PythonApiService class not found!");
}

// Test 5: Check Python configuration
printInfo("\nTest 5: Checking Python configuration...");
$pythonConfigFile = __DIR__ . '/pytho/config.py';
if (file_exists($pythonConfigFile)) {
    printSuccess("Python config.py file exists");
} else {
    printError("Python config.py file not found!");
}

// Summary
printHeader("Test Summary");

$pythonEnvExists = file_exists(__DIR__ . '/pytho/.env');
$laravelEnvExists = file_exists(__DIR__ . '/Laravel/.env');
$configExists = file_exists(__DIR__ . '/Laravel/config/python.php');
$serviceExists = file_exists(__DIR__ . '/Laravel/app/Services/PythonApiService.php');

if ($pythonEnvExists && $laravelEnvExists && $configExists && $serviceExists) {
    printSuccess("All configuration files are in place");
    printInfo("\nNext steps:");
    printInfo("1. Start Python service: cd pytho && docker-compose up -d");
    printInfo("2. Start Laravel: cd Laravel && php artisan serve");
    printInfo("3. Run Laravel tests: cd Laravel && php artisan test --filter=PythonIntegrationTest");
} else {
    printError("Some configuration files are missing");
    printInfo("\nPlease complete the setup:");
    if (!$pythonEnvExists) printInfo("- Copy pytho/.env.example to pytho/.env");
    if (!$laravelEnvExists) printInfo("- Copy Laravel/.env.example to Laravel/.env");
}

echo "\n";