<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CreatorScheduledPost;
use App\Models\User;
use App\Models\SocialAccount;
use App\Models\Tenant;

class TestCreatorScheduledPost extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:creator-scheduled-post';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test creating a creator scheduled post';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Get or create a tenant
        $tenant = Tenant::first();
        
        if (!$tenant) {
            $this->info('Creating a test tenant...');
            $tenant = Tenant::create([
                'name' => 'Test Tenant',
                'domain' => 'test-domain',
            ]);
        }
        
        // Get or create a user (creator)
        $user = User::first();
        
        if (!$user) {
            $this->info('Creating a test user...');
            $user = User::create([
                'name' => 'Test Creator',
                'email' => 'test.creator@example.com',
                'password' => bcrypt('password'),
                'tenant_id' => $tenant->id,
            ]);
        }
        
        // Get or create a social account for this user
        $socialAccount = SocialAccount::where('user_id', $user->id)->first();
        
        if (!$socialAccount) {
            $this->info('Creating a test social account...');
            $socialAccount = SocialAccount::create([
                'user_id' => $user->id,
                'provider' => 'instagram',
                'provider_user_id' => 'test_provider_id',
                'access_token' => 'test_access_token',
                'token_expires_at' => now()->addDays(30),
                'tenant_id' => $tenant->id,
                'meta' => [
                    'username' => 'test_user',
                ],
            ]);
        }
        
        // Create a test scheduled post
        $scheduledPost = CreatorScheduledPost::create([
            'user_id' => $user->id,
            'social_account_id' => $socialAccount->id,
            'caption' => 'This is a test post from the console command',
            'media_refs' => [
                [
                    'url' => 'https://example.com/image.jpg',
                    'type' => 'image'
                ]
            ],
            'publish_time' => now()->addHour(),
            'timezone' => 'Asia/Calcutta',
            'status' => 'scheduled',
            'retry_policy' => json_encode([
                'max_attempts' => 3,
                'backoff' => 'exponential',
                'base_delay' => 60,
                'max_delay' => 86400,
                'multiplier' => 2
            ]),
            'idempotency_key' => \Illuminate\Support\Str::uuid(),
            'post_visibility' => 'public',
            'auto_reply_enabled' => false,
        ]);
        
        $this->info('Created scheduled post with ID: ' . $scheduledPost->id);
        $this->info('Publish time: ' . $scheduledPost->publish_time);
        $this->info('Timezone: ' . $scheduledPost->timezone);
    }
}