<?php

namespace App\Http\Controllers\Agency;

use App\Http\Controllers\Controller;
use App\Models\AiJob;
use App\Models\Proposal;
use App\Models\ProposalRequest;
use App\Jobs\GenerateProposalJob;
use App\Models\ProposalSection;
use App\Models\ProposalSectionAudit;
use App\Models\User;
use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;

class AiProposalController extends Controller
{
    /**
     * Show the form for creating a new AI proposal request for an assigned brand.
     *
     * @param  int|null  $brandId
     * @return \Illuminate\Http\Response
     */
    public function create($brandId = null)
    {
        // Get brands assigned to this agency
        $assignedBrands = Auth::user()->brands;
        
        // If no brands are assigned, show error
        if ($assignedBrands->isEmpty()) {
            return response()->view('errors.no-assigned-brands', [], 403);
        }
        
        $selectedBrand = null;
        if ($brandId) {
            $selectedBrand = $assignedBrands->find($brandId);
            if (!$selectedBrand) {
                return response()->view('errors.brand-not-assigned', [], 403);
            }
        }
        
        // Get all creators for the dropdown
        $creators = User::whereHas('roles', function ($q) {
            $q->where('name', 'creator');
        })->get();
        
        return view('agency.ai-proposals.create', compact('assignedBrands', 'selectedBrand', 'creators'));
    }
    
    /**
     * Store a new AI proposal request for an assigned brand.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Get brands assigned to this agency
        $assignedBrands = Auth::user()->brands;
        
        // If no brands are assigned, show error
        if ($assignedBrands->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'You have no assigned brands.',
            ], 403);
        }
        
        // Validate inputs
        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'creator_id' => 'nullable|exists:users,id',
            'campaign_brief' => 'required|string|max:5000',
            'tone' => 'required|string|max:255',
            'budget_range' => 'required|array|size:2',
            'budget_range.*' => 'numeric|min:0',
            'deliverables' => 'required|array',
            'deliverables.*' => 'string|max:255',
        ]);
        
        // Check if the brand is assigned to this agency
        $brand = $assignedBrands->find($validated['brand_id']);
        if (!$brand) {
            return response()->json([
                'success' => false,
                'message' => 'You do not have permission to create proposals for this brand.',
            ], 403);
        }
        
        // Rate limiting - 10 requests per hour per agency
        $throttleKey = 'ai-proposal-request|agency|' . Auth::id();
        if (RateLimiter::tooManyAttempts($throttleKey, 10)) {
            $retryAfter = RateLimiter::availableIn($throttleKey);
            return response()->json([
                'success' => false,
                'message' => 'Too many requests. Please try again in ' . $retryAfter . ' seconds.',
            ], 429);
        }
        
        // Sanitize PII before sending to AI service
        $sanitizedBrief = $this->sanitizePii($validated['campaign_brief']);
        
        // Create proposal request with agency metadata
        $proposalRequest = ProposalRequest::create([
            'brand_id' => $brand->id,
            'creator_id' => $validated['creator_id'] ?? null,
            'campaign_brief' => $sanitizedBrief,
            'tone' => $validated['tone'],
            'budget_range' => $validated['budget_range'],
            'deliverables' => $validated['deliverables'],
            'status' => 'pending',
        ]);
        
        // Enqueue AI job
        $aiJob = $this->enqueueAiJob($brand->id, $validated['creator_id'] ?? null, [
            'campaign_brief' => $sanitizedBrief,
            'tone' => $validated['tone'],
            'budget_range' => $validated['budget_range'],
            'deliverables' => $validated['deliverables'],
            'brand_name' => $brand->name,
            'brand_industry' => $brand->settings['industry'] ?? 'general',
        ]);
        
        // Link proposal request with AI job
        $proposalRequest->update(['ai_job_id' => $aiJob->id]);
        
        // Increment rate limiter with 1 hour decay
        RateLimiter::hit($throttleKey, 3600);
        
        // Log the agency action
        \App\Models\AuditLog::create([
            'actor_id' => Auth::id(),
            'action' => 'generate_ai_proposal',
            'target_type' => 'AiJob',
            'target_id' => $aiJob->id,
            'meta' => [
                'brand_id' => $brand->id,
                'creator_id' => $validated['creator_id'] ?? null,
                'agency_created' => true,
            ],
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'AI proposal generation started! The proposal will require client approval before sending to creators.',
            'job_id' => $aiJob->job_id,
        ]);
    }
    
    /**
     * Display the generated proposal.
     *
     * @param  string  $jobId
     * @return \Illuminate\Http\Response
     */
    public function show($jobId)
    {
        // Get brands assigned to this agency
        $assignedBrands = Auth::user()->brands;
        
        // If no brands are assigned, show error
        if ($assignedBrands->isEmpty()) {
            return response()->view('errors.no-assigned-brands', [], 403);
        }
        
        $aiJob = AiJob::where('job_id', $jobId)
            ->whereIn('brand_id', $assignedBrands->pluck('id'))
            ->firstOrFail();
        
        return view('agency.ai-proposals.show', compact('aiJob'));
    }
    
    /**
     * Update the generated proposal (save draft or request approval).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $jobId
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $jobId)
    {
        // Get brands assigned to this agency
        $assignedBrands = Auth::user()->brands;
        
        // If no brands are assigned, show error
        if ($assignedBrands->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'You have no assigned brands.',
            ], 403);
        }
        
        $aiJob = AiJob::where('job_id', $jobId)
            ->whereIn('brand_id', $assignedBrands->pluck('id'))
            ->firstOrFail();
        
        $validated = $request->validate([
            'action' => 'required|in:save_draft,request_approval',
            'title' => 'required_if:action,request_approval|string|max:255',
            'sections' => 'required_if:action,request_approval|array',
            'sections.*.title' => 'required_if:action,request_approval|string|max:255',
            'sections.*.content' => 'required_if:action,request_approval|string',
            'sections.*.order' => 'required_if:action,request_approval|integer',
            'price_estimate' => 'required_if:action,request_approval|numeric|min:0',
        ]);
        
        switch ($validated['action']) {
            case 'save_draft':
                // Just save the current state as a draft
                return response()->json([
                    'success' => true,
                    'message' => 'Draft saved successfully!',
                ]);
            
            case 'request_approval':
                // Create a proposal from the AI-generated content
                $proposal = Proposal::create([
                    'brand_id' => $aiJob->brand_id,
                    'creator_id' => $aiJob->creator_id,
                    'title' => $validated['title'],
                    'message' => 'AI-generated proposal by agency',
                    'type' => 'Collaboration',
                    'budget' => $validated['price_estimate'],
                    'status' => 'pending_approval', // Agency proposals require brand admin approval
                    'ai_job_id' => $aiJob->id,
                ]);
                
                // Create proposal sections
                foreach ($validated['sections'] as $sectionData) {
                    $proposalSection = ProposalSection::create([
                        'proposal_id' => $proposal->id,
                        'title' => $sectionData['title'],
                        'content' => $sectionData['content'],
                        'order' => $sectionData['order'],
                        'is_editable' => true,
                        'edited_by' => Auth::id(),
                        'edited_at' => now(),
                    ]);
                    
                    // Create audit trail entry
                    ProposalSectionAudit::create([
                        'proposal_section_id' => $proposalSection->id,
                        'user_id' => Auth::id(),
                        'action' => 'agency_create',
                        'old_content' => null,
                        'new_content' => $sectionData['content'],
                    ]);
                }
                
                // Update AI job status
                $aiJob->update([
                    'status' => 'accepted',
                    'proposal_id' => $proposal->id,
                    'completed_at' => now(),
                ]);
                
                // Log the agency action
                \App\Models\AuditLog::create([
                    'actor_id' => Auth::id(),
                    'action' => 'request_proposal_approval',
                    'target_type' => 'Proposal',
                    'target_id' => $proposal->id,
                    'meta' => [
                        'ai_job_id' => $aiJob->id,
                        'agency_created' => true,
                    ],
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Proposal created and sent for client approval!',
                    'redirect' => route('agency.dashboard'),
                ]);
        }
    }
    
    /**
     * Enqueue an AI job for proposal generation.
     *
     * @param  int  $brandId
     * @param  int|null  $creatorId
     * @param  array  $payload
     * @return \App\Models\AiJob
     */
    protected function enqueueAiJob($brandId, $creatorId, $payload)
    {
        $jobId = (string) Str::uuid();
        
        $aiJob = AiJob::create([
            'job_id' => $jobId,
            'type' => 'generate_proposal',
            'status' => 'queued',
            'brand_id' => $brandId,
            'creator_id' => $creatorId,
            'result_text' => null,
            'result_meta' => null,
            'provider' => null,
        ]);
        
        // Dispatch the actual job
        GenerateProposalJob::dispatch($aiJob->id, $payload);
        
        return $aiJob;
    }
    
    /**
     * Sanitize PII from text before sending to AI service.
     *
     * @param  string  $text
     * @return string
     */
    protected function sanitizePii($text)
    {
        // Remove email addresses
        $text = preg_replace('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', '[EMAIL]', $text);
        
        // Remove phone numbers (various formats)
        $text = preg_replace('/(\+?\d{1,3}[-.\s]?)?\(?\d{3}\)?[-.\s]?\d{3}[-.\s]?\d{4}/', '[PHONE]', $text);
        
        // Remove URLs
        $text = preg_replace('/https?:\/\/(www\.)?[-a-zA-Z0-9@:%._\+~#=]{1,256}\.[a-zA-Z0-9()]{1,6}\b([-a-zA-Z0-9()@:%_\+.~#?&//=]*)/', '[URL]', $text);
        
        // Remove credit card numbers (simple pattern)
        $text = preg_replace('/\b\d{4}[-\s]?\d{4}[-\s]?\d{4}[-\s]?\d{4}\b/', '[CREDIT CARD]', $text);
        
        // Remove social security numbers
        $text = preg_replace('/\b\d{3}-\d{2}-\d{4}\b/', '[SSN]', $text);
        
        // Remove IP addresses
        $text = preg_replace('/\b\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}\b/', '[IP ADDRESS]', $text);
        
        return $text;
    }
}