<?php

namespace App\Http\Controllers\Agency;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Campaign;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CampaignController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Get campaigns for the current agency
        $campaigns = Campaign::where('agency_id', Auth::id())->with('brand')->get();
        
        // Get brands linked to this agency
        $brands = Auth::user()->brands;
        
        return view('agency.campaigns.index', compact('campaigns', 'brands'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get brands linked to this agency
        $brands = Auth::user()->brands;
        
        return view('agency.campaigns.create', compact('brands'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'budget' => 'required|numeric|min:0',
            'status' => 'required|string|max:255',
        ]);

        Campaign::create([
            'agency_id' => Auth::id(),
            'brand_id' => $request->brand_id,
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'budget' => $request->budget,
            'status' => $request->status,
        ]);

        return redirect()->route('agency.campaigns.index')->with('success', 'Campaign created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Campaign $campaign)
    {
        // Ensure the campaign belongs to the current agency
        if ($campaign->agency_id !== Auth::id()) {
            abort(403);
        }
        
        // Get brands linked to this agency
        $brands = Auth::user()->brands;
        
        return view('agency.campaigns.edit', compact('campaign', 'brands'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Campaign $campaign)
    {
        // Ensure the campaign belongs to the current agency
        if ($campaign->agency_id !== Auth::id()) {
            abort(403);
        }
        
        $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'budget' => 'required|numeric|min:0',
            'status' => 'required|string|max:255',
        ]);

        $campaign->update([
            'brand_id' => $request->brand_id,
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'budget' => $request->budget,
            'status' => $request->status,
        ]);

        return redirect()->route('agency.campaigns.index')->with('success', 'Campaign updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Campaign $campaign)
    {
        // Ensure the campaign belongs to the current agency
        if ($campaign->agency_id !== Auth::id()) {
            abort(403);
        }
        
        $campaign->delete();

        return redirect()->route('agency.campaigns.index')->with('success', 'Campaign deleted successfully.');
    }
}