<?php

namespace App\Http\Controllers\Agency;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SidebarController extends Controller
{
    /**
     * Get badge counts for agency sidebar navigation
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBadges(Request $request)
    {
        $user = $request->user();
        $agencyId = $user->tenant_id; // Assuming agency uses tenant_id

        // Get total managed brands count
        $totalBrands = DB::table('agency_brand')
            ->where('agency_id', $agencyId)
            ->count();

        // Get pending proposals count (across all managed brands)
        $pendingProposals = DB::table('proposals')
            ->where('agency_id', $agencyId)
            ->whereIn('status', ['draft', 'sent', 'negotiation'])
            ->count();

        // Get active campaigns count (across all managed brands)
        $activeCampaigns = DB::table('campaigns')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->where('campaigns.status', 'active')
            ->count();

        // Get unread messages count (from brands and creators)
        $unreadMessages = DB::table('message_threads')
            ->join('message_thread_participants', 'message_threads.id', '=', 'message_thread_participants.thread_id')
            ->join('messages', 'message_threads.id', '=', 'messages.thread_id')
            ->where('message_thread_participants.user_id', $user->id)
            ->where('messages.sender_id', '!=', $user->id)
            ->whereNull('messages.read_at')
            ->distinct('message_threads.id')
            ->count('message_threads.id');

        // Get notifications count (unread)
        $notifications = $user->unreadNotifications()->count();

        return response()->json([
            'success' => true,
            'badges' => [
                'totalBrands' => $totalBrands,
                'pendingProposals' => $pendingProposals,
                'activeCampaigns' => $activeCampaigns,
                'unreadMessages' => $unreadMessages,
                'notifications' => $notifications,
            ]
        ]);
    }

    /**
     * Get quick search results for agency
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function quickSearch(Request $request)
    {
        $query = $request->input('query', '');
        $user = $request->user();
        $agencyId = $user->tenant_id;

        if (strlen($query) < 2) {
            return response()->json([
                'success' => true,
                'results' => []
            ]);
        }

        $results = [];

        // Search managed brands
        $brands = DB::table('brands')
            ->join('agency_brand', 'brands.id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->where('brands.name', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['brands.id', 'brands.name', 'brands.industry']);

        foreach ($brands as $brand) {
            $results[] = [
                'name' => $brand->name,
                'category' => 'Brand',
                'url' => route('agency.brands.show', $brand->id),
                'meta' => $brand->industry ?? 'Managed Brand'
            ];
        }

        // Search campaigns (across all managed brands)
        $campaigns = DB::table('campaigns')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->where('campaigns.name', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['campaigns.id', 'campaigns.name', 'campaigns.status']);

        foreach ($campaigns as $campaign) {
            $results[] = [
                'name' => $campaign->name,
                'category' => 'Campaign',
                'url' => route('agency.campaigns.show', $campaign->id),
                'meta' => "Status: {$campaign->status}"
            ];
        }

        // Search creators
        $creators = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'creator')
            ->where('users.name', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['users.id', 'users.name', 'users.email']);

        foreach ($creators as $creator) {
            $results[] = [
                'name' => $creator->name,
                'category' => 'Creator',
                'url' => route('agency.creators.show', $creator->id),
                'meta' => $creator->email
            ];
        }

        // Search proposals (across all managed brands)
        $proposals = DB::table('proposals')
            ->where('agency_id', $agencyId)
            ->where('title', 'LIKE', "%{$query}%")
            ->limit(5)
            ->get(['id', 'title', 'status']);

        foreach ($proposals as $proposal) {
            $results[] = [
                'name' => $proposal->title,
                'category' => 'Proposal',
                'url' => route('agency.proposals.show', $proposal->id),
                'meta' => "Status: {$proposal->status}"
            ];
        }

        return response()->json([
            'success' => true,
            'results' => $results
        ]);
    }

    /**
     * Get agency dashboard KPIs
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDashboardKPIs(Request $request)
    {
        $user = $request->user();
        $agencyId = $user->tenant_id;

        // Total managed brands
        $totalBrands = DB::table('agency_brand')
            ->where('agency_id', $agencyId)
            ->count();

        // Active campaigns
        $activeCampaigns = DB::table('campaigns')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->where('campaigns.status', 'active')
            ->count();

        // Pending proposals
        $pendingProposals = DB::table('proposals')
            ->where('agency_id', $agencyId)
            ->whereIn('status', ['draft', 'sent', 'negotiation'])
            ->count();

        // Revenue share (sum of all payments)
        $revenueShare = DB::table('payments')
            ->join('proposals', 'payments.proposal_id', '=', 'proposals.id')
            ->where('proposals.agency_id', $agencyId)
            ->where('payments.status', 'completed')
            ->sum('payments.amount');

        // Total creators engaged
        $totalCreators = DB::table('campaign_creators')
            ->join('campaigns', 'campaign_creators.campaign_id', '=', 'campaigns.id')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->distinct('campaign_creators.creator_id')
            ->count('campaign_creators.creator_id');

        // Completed campaigns
        $completedCampaigns = DB::table('campaigns')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->where('agency_brand.agency_id', $agencyId)
            ->where('campaigns.status', 'completed')
            ->count();

        return response()->json([
            'success' => true,
            'kpis' => [
                'totalBrands' => $totalBrands,
                'activeCampaigns' => $activeCampaigns,
                'pendingProposals' => $pendingProposals,
                'revenueShare' => number_format($revenueShare, 2),
                'totalCreators' => $totalCreators,
                'completedCampaigns' => $completedCampaigns,
            ]
        ]);
    }

    /**
     * Get recent activity for agency dashboard
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRecentActivity(Request $request)
    {
        $user = $request->user();
        $agencyId = $user->tenant_id;

        $activities = [];

        // Recent campaigns
        $recentCampaigns = DB::table('campaigns')
            ->join('agency_brand', 'campaigns.brand_id', '=', 'agency_brand.brand_id')
            ->join('brands', 'campaigns.brand_id', '=', 'brands.id')
            ->where('agency_brand.agency_id', $agencyId)
            ->orderBy('campaigns.created_at', 'desc')
            ->limit(5)
            ->get(['campaigns.id', 'campaigns.name', 'campaigns.status', 'campaigns.created_at', 'brands.name as brand_name']);

        foreach ($recentCampaigns as $campaign) {
            $activities[] = [
                'type' => 'campaign',
                'title' => "New campaign: {$campaign->name}",
                'description' => "Brand: {$campaign->brand_name}",
                'timestamp' => $campaign->created_at,
                'url' => route('agency.campaigns.show', $campaign->id)
            ];
        }

        // Recent proposals
        $recentProposals = DB::table('proposals')
            ->join('brands', 'proposals.brand_id', '=', 'brands.id')
            ->where('proposals.agency_id', $agencyId)
            ->orderBy('proposals.created_at', 'desc')
            ->limit(5)
            ->get(['proposals.id', 'proposals.title', 'proposals.status', 'proposals.created_at', 'brands.name as brand_name']);

        foreach ($recentProposals as $proposal) {
            $activities[] = [
                'type' => 'proposal',
                'title' => "Proposal: {$proposal->title}",
                'description' => "Brand: {$proposal->brand_name} - Status: {$proposal->status}",
                'timestamp' => $proposal->created_at,
                'url' => route('agency.proposals.show', $proposal->id)
            ];
        }

        // Sort by timestamp
        usort($activities, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });

        return response()->json([
            'success' => true,
            'activities' => array_slice($activities, 0, 10)
        ]);
    }
}