<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\RegistrationRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;

class SignupRequestController extends Controller
{
    /**
     * Show the form for requesting registration.
     */
    public function create()
    {
        return view('signup.request');
    }

    /**
     * Store a new registration request.
     */
    public function store(Request $request)
    {
        // Rate limiting
        $throttleKey = 'signup-request|' . $request->ip();
        
        if (RateLimiter::tooManyAttempts($throttleKey, 5)) {
            return back()->withErrors([
                'email' => 'Too many requests. Please try again later.'
            ]);
        }
        
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:registration_requests'],
            'requested_role' => ['required', 'in:brand_admin,creator,agency_admin'],
            'company_name' => ['nullable', 'string', 'max:255'],
            'website' => ['nullable', 'string', 'max:255'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);
        
        // Create registration request
        $registrationRequest = RegistrationRequest::create([
            'name' => $request->name,
            'email' => $request->email,
            'requested_role' => $request->requested_role,
            'meta' => [
                'company_name' => $request->company_name,
                'website' => $request->website,
                'notes' => $request->notes,
            ],
            'status' => 'pending',
        ]);
        
        // Log the request creation
        \App\Models\AuditLog::create([
            'actor_id' => null, // No user yet
            'action' => 'registration_request_created',
            'target_type' => 'registration_request',
            'target_id' => $registrationRequest->id,
            'meta' => [
                'email' => $request->email,
                'requested_role' => $request->requested_role,
            ]
        ]);
        
        // Increment rate limiter
        RateLimiter::hit($throttleKey);
        
        return redirect()->back()->with('status', 'Your request has been submitted successfully. We will review it and get back to you soon.');
    }
}