<?php

namespace App\Http\Controllers\Brand;

use App\Http\Controllers\Controller;
use App\Models\ReelRequest;
use App\Models\ReelInput;
use App\Models\SocialPost;
use App\Models\ScheduledReel;
use App\Http\Requests\StoreReelRequest;
use App\Http\Requests\StoreScheduledReelRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ReelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $reels = ReelRequest::where('brand_id', auth()->user()->brand_id)
            ->with(['user', 'inputs.sourcePost'])
            ->latest()
            ->paginate(10);

        // Get scheduled reels
        $scheduledReels = ScheduledReel::where('owner_id', auth()->user()->brand_id)
            ->where('owner_type', 'App\Models\Brand')
            ->with(['poster'])
            ->latest()
            ->paginate(10);

        return view('brand.reels.index', compact('reels', 'scheduledReels'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get social posts for the brand's creators
        $socialPosts = SocialPost::whereHas('socialAccount.user', function ($query) {
            $query->where('brand_id', auth()->user()->brand_id);
        })->with('socialAccount.user')->get();

        return view('brand.reels.create', compact('socialPosts'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreReelRequest $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                // Create the reel request
                $reelRequest = ReelRequest::create([
                    'user_id' => auth()->id(),
                    'brand_id' => auth()->user()->brand_id,
                    'template' => $request->template,
                    'options' => $request->options ?? [],
                    'status' => 'queued'
                ]);

                // Create reel inputs
                if ($request->has('source_posts')) {
                    foreach ($request->source_posts as $postId) {
                        $post = SocialPost::find($postId);
                        
                        // Verify the post belongs to a creator of this brand
                        if ($post && $post->socialAccount && $post->socialAccount->user->brand_id == auth()->user()->brand_id) {
                            ReelInput::create([
                                'reel_request_id' => $reelRequest->id,
                                'source_post_id' => $postId,
                                'provider' => $post->provider,
                                'source_url' => $post->media_url,
                                'meta' => [
                                    'caption' => $post->caption,
                                    'created_at' => $post->created_at
                                ]
                            ]);
                        }
                    }
                }

                // Dispatch job to process the reel
                \App\Jobs\SendReelToProcessor::dispatch($reelRequest);

                return redirect()->route('brand.reels.show', $reelRequest)
                    ->with('success', 'Reel request created successfully and is being processed.');
            });
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to create reel request: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Store a newly created scheduled reel in storage.
     */
    public function storeScheduled(StoreScheduledReelRequest $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                // Create the scheduled reel
                $scheduledReel = ScheduledReel::create([
                    'owner_id' => auth()->user()->brand_id,
                    'owner_type' => 'App\Models\Brand',
                    'template' => $request->template,
                    'options' => $request->options ?? [],
                    'source_inputs' => $request->source_inputs,
                    'scheduled_at' => $request->scheduled_at,
                    'timezone' => $request->timezone,
                    'status' => $request->scheduled_at ? 'scheduled' : 'draft',
                    'created_by' => auth()->id(),
                ]);

                // Send notification
                auth()->user()->notify(new \App\Notifications\ReelScheduled($scheduledReel));

                return redirect()->route('brand.reels.show', $scheduledReel)
                    ->with('success', 'Reel scheduled successfully.');
            });
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to schedule reel: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(ReelRequest $reel)
    {
        // Ensure the user can only view their own reels or reels from their brand
        if ($reel->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        $reel->load(['inputs.sourcePost.user']);

        return view('brand.reels.show', compact('reel'));
    }

    /**
     * Display the specified scheduled reel.
     */
    public function showScheduled(ScheduledReel $reel)
    {
        // Ensure the user can only view their own scheduled reels or reels from their brand
        if ($reel->owner_id != auth()->user()->brand_id || $reel->owner_type !== 'App\Models\Brand') {
            abort(403);
        }

        $reel->load(['postAttempts', 'createdBy']);

        return view('brand.reels.show-scheduled', compact('reel'));
    }

    /**
     * Post a scheduled reel immediately.
     */
    public function postNow(ScheduledReel $reel)
    {
        // Ensure the user can only post their own scheduled reels or reels from their brand
        if ($reel->owner_id != auth()->user()->brand_id || $reel->owner_type !== 'App\Models\Brand') {
            abort(403);
        }

        try {
            // Dispatch job to process the scheduled reel immediately
            \App\Jobs\ProcessScheduledReel::dispatch($reel);
            
            // Mark as queued
            $reel->markQueued();

            return back()->with('success', 'Reel is being posted now.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to post reel: ' . $e->getMessage());
        }
    }

    /**
     * Reschedule a reel.
     */
    public function reschedule(ScheduledReel $reel, Request $request)
    {
        // Ensure the user can only reschedule their own scheduled reels or reels from their brand
        if ($reel->owner_id != auth()->user()->brand_id || $reel->owner_type !== 'App\Models\Brand') {
            abort(403);
        }

        $request->validate([
            'scheduled_at' => 'required|date',
            'timezone' => 'required|string|timezone',
        ]);

        try {
            $reel->update([
                'scheduled_at' => $request->scheduled_at,
                'timezone' => $request->timezone,
                'status' => 'scheduled',
            ]);

            return back()->with('success', 'Reel has been rescheduled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to reschedule reel: ' . $e->getMessage());
        }
    }

    /**
     * Cancel a scheduled reel.
     */
    public function cancel(ScheduledReel $reel)
    {
        // Ensure the user can only cancel their own scheduled reels or reels from their brand
        if ($reel->owner_id != auth()->user()->brand_id || $reel->owner_type !== 'App\Models\Brand') {
            abort(403);
        }

        try {
            $reel->markCancelled();

            return back()->with('success', 'Reel has been cancelled.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel reel: ' . $e->getMessage());
        }
    }

    /**
     * Retry processing a failed reel.
     */
    public function retry(ReelRequest $reel)
    {
        // Ensure the user can only retry their own reels or reels from their brand
        if ($reel->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        if ($reel->status != 'failed') {
            return back()->with('error', 'Only failed reels can be retried.');
        }

        try {
            $reel->update([
                'status' => 'queued',
                'error_message' => null,
                'job_id' => null,
                'result_path' => null,
                'duration' => null
            ]);

            // Dispatch job to process the reel
            \App\Jobs\SendReelToProcessor::dispatch($reel);

            return back()->with('success', 'Reel processing has been restarted.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to retry reel processing: ' . $e->getMessage());
        }
    }

    /**
     * Download the generated reel.
     */
    public function download(ReelRequest $reel)
    {
        // Ensure the user can only download their own reels or reels from their brand
        if ($reel->brand_id != auth()->user()->brand_id) {
            abort(403);
        }

        if (!$reel->isReady()) {
            return back()->with('error', 'Reel is not ready for download yet.');
        }

        if (!$reel->result_path || !Storage::disk('public')->exists($reel->result_path)) {
            return back()->with('error', 'Reel file not found.');
        }

        return Storage::disk('public')->download($reel->result_path);
    }
}